<?php

class GistMag_Blog_Generator_Admin {
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }

    public function add_admin_menu() {
        add_menu_page(
            __('GistMag Blog Meta Generator', 'gistmag-blog-generator'),
            __('GistMag Blog Meta', 'gistmag-blog-generator'),
            'manage_options',
            'gistmag-blog-generator',
            array($this, 'render_settings_page'),
            'dashicons-edit',
            31
        );
    }

    public function register_settings() {
        // phpcs:disable PluginCheck.CodeAnalysis.SettingSanitization.register_settingMissing
        // All settings have sanitize_callback defined in the array parameter
        register_setting(
            'gistmag_blog_generator_settings',
            'gistmag_blog_generator_api_key',
            array(
                'sanitize_callback' => 'sanitize_text_field',
            )
        );

        register_setting(
            'gistmag_blog_generator_settings',
            'gistmag_blog_generator_enabled',
            array(
                'sanitize_callback' => 'rest_sanitize_boolean',
            )
        );

        register_setting(
            'gistmag_blog_generator_settings',
            'gistmag_blog_generator_default_image_style',
            array(
                'sanitize_callback' => array($this, 'sanitize_image_style'),
            )
        );

        register_setting(
            'gistmag_blog_generator_settings',
            'gistmag_blog_generator_auto_apply',
            array(
                'sanitize_callback' => 'rest_sanitize_boolean',
            )
        );
        // phpcs:enable
    }

    /**
     * Sanitize image style value.
     *
     * @param string $value Image style value.
     * @return string
     */
    public function sanitize_image_style($value) {
        $value = sanitize_text_field($value);
        $allowed = array('realistic', 'illustration', 'abstract', 'minimalist', 'photographic');
        if (!in_array($value, $allowed, true)) {
            return 'realistic';
        }
        return $value;
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $api = new GistMag_Blog_Generator_API();
        $api_key = get_option('gistmag_blog_generator_api_key', '');
        $usage_stats = null;

        // Handle usage stats check with nonce verification.
        if (
            ! empty( $api_key )
            && isset( $_POST['check_usage'], $_POST['gistmag_blog_generator_check_usage_nonce'] )
        ) {
            $raw_nonce = wp_unslash( $_POST['gistmag_blog_generator_check_usage_nonce'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $nonce     = sanitize_text_field( $raw_nonce );

            if ( wp_verify_nonce( $nonce, 'gistmag_blog_generator_check_usage' ) ) {
                $usage_stats = $api->get_usage_stats( $api_key );
            }
        }

        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('gistmag_blog_generator_settings');
                do_settings_sections('gistmag_blog_generator_settings');
                ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="gistmag_blog_generator_api_key"><?php esc_html_e('API Key', 'gistmag-blog-generator'); ?></label>
                        </th>
                        <td>
                            <input type="text" 
                                   id="gistmag_blog_generator_api_key" 
                                   name="gistmag_blog_generator_api_key" 
                                   value="<?php echo esc_attr($api_key); ?>" 
                                   class="regular-text"
                                   placeholder="atg_...">
                            <p class="description">
                                <?php esc_html_e('Enter your GistMag API key. Get one at', 'gistmag-blog-generator'); ?> 
                                <a href="https://gistmag.co.uk/dashboard/api-key" target="_blank">gistmag.co.uk</a>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="gistmag_blog_generator_enabled"><?php esc_html_e('Enable Plugin', 'gistmag-blog-generator'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" 
                                   id="gistmag_blog_generator_enabled" 
                                   name="gistmag_blog_generator_enabled" 
                                   value="1" 
                                   <?php checked(get_option('gistmag_blog_generator_enabled', true)); ?>>
                            <label for="gistmag_blog_generator_enabled">
                                <?php esc_html_e('Enable blog meta generator functionality', 'gistmag-blog-generator'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="gistmag_blog_generator_default_image_style"><?php esc_html_e('Default Image Style', 'gistmag-blog-generator'); ?></label>
                        </th>
                        <td>
                            <select id="gistmag_blog_generator_default_image_style" name="gistmag_blog_generator_default_image_style">
                                <option value="realistic" <?php selected(get_option('gistmag_blog_generator_default_image_style', 'realistic'), 'realistic'); ?>><?php esc_html_e('Realistic', 'gistmag-blog-generator'); ?></option>
                                <option value="illustration" <?php selected(get_option('gistmag_blog_generator_default_image_style', 'realistic'), 'illustration'); ?>><?php esc_html_e('Illustration', 'gistmag-blog-generator'); ?></option>
                                <option value="abstract" <?php selected(get_option('gistmag_blog_generator_default_image_style', 'realistic'), 'abstract'); ?>><?php esc_html_e('Abstract', 'gistmag-blog-generator'); ?></option>
                                <option value="minimalist" <?php selected(get_option('gistmag_blog_generator_default_image_style', 'realistic'), 'minimalist'); ?>><?php esc_html_e('Minimalist', 'gistmag-blog-generator'); ?></option>
                                <option value="photographic" <?php selected(get_option('gistmag_blog_generator_default_image_style', 'realistic'), 'photographic'); ?>><?php esc_html_e('Photographic', 'gistmag-blog-generator'); ?></option>
                            </select>
                            <p class="description">
                                <?php esc_html_e('Default image style for generated images', 'gistmag-blog-generator'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="gistmag_blog_generator_auto_apply"><?php esc_html_e('Auto-Apply Results', 'gistmag-blog-generator'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox" 
                                   id="gistmag_blog_generator_auto_apply" 
                                   name="gistmag_blog_generator_auto_apply" 
                                   value="1" 
                                   <?php checked(get_option('gistmag_blog_generator_auto_apply', false)); ?>>
                            <label for="gistmag_blog_generator_auto_apply">
                                <?php esc_html_e('Automatically apply generated excerpt, category, and tags to posts', 'gistmag-blog-generator'); ?>
                            </label>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>

            <?php if (!empty($api_key)) : ?>
                <hr>
                <h2><?php esc_html_e('Usage Statistics', 'gistmag-blog-generator'); ?></h2>
                <form method="post" action="">
                    <?php wp_nonce_field('gistmag_blog_generator_check_usage', 'gistmag_blog_generator_check_usage_nonce'); ?>
                    <p>
                        <input type="submit" name="check_usage" class="button" value="<?php esc_attr_e('Check Usage', 'gistmag-blog-generator'); ?>">
                    </p>
                </form>

                <?php if ($usage_stats && !is_wp_error($usage_stats)) : ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Current Month Usage', 'gistmag-blog-generator'); ?></th>
                            <td><?php echo esc_html($usage_stats['current_month'] ?? 0); ?></td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Monthly Limit', 'gistmag-blog-generator'); ?></th>
                            <td><?php echo esc_html($usage_stats['monthly_limit'] ?? 0); ?></td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Remaining', 'gistmag-blog-generator'); ?></th>
                            <td><?php echo esc_html($usage_stats['remaining'] ?? 0); ?></td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Credits Used', 'gistmag-blog-generator'); ?></th>
                            <td><?php echo esc_html($usage_stats['credits_used'] ?? 0); ?></td>
                        </tr>
                    </table>
                <?php elseif ($usage_stats && is_wp_error($usage_stats)) : ?>
                    <div class="notice notice-error">
                        <p><?php echo esc_html($usage_stats->get_error_message()); ?></p>
                    </div>
                <?php endif; ?>
            <?php endif; ?>

            <hr>
            <h2><?php esc_html_e('How to Use', 'gistmag-blog-generator'); ?></h2>
            <ol>
                <li><?php esc_html_e('Enter your GistMag API key above', 'gistmag-blog-generator'); ?></li>
                <li><?php esc_html_e('Go to any post editor page', 'gistmag-blog-generator'); ?></li>
                <li><?php esc_html_e('Find the "GistMag Blog Meta Generator" meta box in the sidebar', 'gistmag-blog-generator'); ?></li>
                <li><?php esc_html_e('Enter your post title and optionally the content', 'gistmag-blog-generator'); ?></li>
                <li><?php esc_html_e('Click "Generate Meta Elements" to generate images, excerpt, category, and tags', 'gistmag-blog-generator'); ?></li>
                <li><?php esc_html_e('Review and apply the generated content to your post', 'gistmag-blog-generator'); ?></li>
            </ol>
            <p>
                <strong><?php esc_html_e('Cost:', 'gistmag-blog-generator'); ?></strong> 
                <?php esc_html_e('3 credits per generation', 'gistmag-blog-generator'); ?>
            </p>
        </div>
        <?php
    }
}

