<?php

class GistMag_Blog_Generator_API {
    private $api_base_url;

    public function __construct() {
        $this->api_base_url = GISTMAG_BLOG_GENERATOR_API_BASE_URL;
    }

    /**
     * Generate blog post elements
     *
     * @param string $title Blog post title
     * @param string $content Optional blog post content
     * @param string $image_style Image style preference
     * @param string $api_key API key
     * @return array|WP_Error Response data or error
     */
    public function generate_blog_elements($title, $content = '', $image_style = 'realistic', $api_key = '') {
        if (empty($api_key)) {
            $api_key = get_option('gistmag_blog_generator_api_key', '');
        }

        if (empty($api_key)) {
            return new WP_Error('no_api_key', __('API key is required', 'gistmag-blog-generator'));
        }

        $url = $this->api_base_url . '/blog/generate';

        $body = array(
            'title' => $title,
            'api_key' => $api_key,
            'image_style' => $image_style,
            'language' => 'en',
            'source' => 'wordpress',
        );

        if (!empty($content)) {
            $body['content'] = $content;
        }

        $response = wp_remote_post($url, array(
            'timeout' => 120, // 2 minutes for image generation
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode($body),
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code !== 200) {
            $error_message = isset($data['message']) ? $data['message'] : __('Unknown error occurred', 'gistmag-blog-generator');
            return new WP_Error('api_error', $error_message, array('status' => $status_code));
        }

        return $data;
    }

    /**
     * Verify API key
     *
     * @param string $api_key API key to verify
     * @return array|WP_Error Response data or error
     */
    public function verify_api_key($api_key) {
        $url = $this->api_base_url . '/verify-key';

        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'api_key' => $api_key,
            )),
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code !== 200) {
            return new WP_Error('invalid_key', __('Invalid API key', 'gistmag-blog-generator'));
        }

        return $data;
    }

    /**
     * Get usage statistics
     *
     * @param string $api_key API key
     * @return array|WP_Error Response data or error
     */
    public function get_usage_stats($api_key) {
        $url = $this->api_base_url . '/usage';

        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode(array(
                'api_key' => $api_key,
            )),
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code !== 200) {
            return new WP_Error('api_error', __('Failed to fetch usage statistics', 'gistmag-blog-generator'));
        }

        return $data;
    }
}

