<?php

class GistMag_Blog_Generator {
    private $api;

    public function __construct() {
        $this->api = new GistMag_Blog_Generator_API();
    }

    public function run() {
        // Enqueue styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_styles'));
        
        // Add meta box to post editor
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        
        // Add AJAX handlers
        add_action('wp_ajax_gistmag_blog_generator_generate', array($this, 'ajax_generate'));
        add_action('wp_ajax_gistmag_blog_generator_verify_key', array($this, 'ajax_verify_key'));
        add_action('wp_ajax_gistmag_blog_generator_apply_category', array($this, 'ajax_apply_category'));
        add_action('wp_ajax_gistmag_blog_generator_apply_tags', array($this, 'ajax_apply_tags'));
        add_action('wp_ajax_gistmag_blog_generator_set_featured_image', array($this, 'ajax_set_featured_image'));
    }

    public function enqueue_styles($hook) {
        // Only load on post edit pages
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }

        wp_enqueue_style(
            'gistmag-blog-generator',
            GISTMAG_BLOG_GENERATOR_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            GISTMAG_BLOG_GENERATOR_VERSION
        );
    }

    public function add_meta_box() {
        $post_types = get_post_types(array('public' => true));
        foreach ($post_types as $post_type) {
            add_meta_box(
                'gistmag_blog_generator',
                __('GistMag Blog Meta Generator', 'gistmag-blog-generator'),
                array($this, 'render_meta_box'),
                $post_type,
                'side',
                'high'
            );
        }
    }

    public function render_meta_box($post) {
        $api_key = get_option('gistmag_blog_generator_api_key', '');
        $image_style = get_option('gistmag_blog_generator_default_image_style', 'realistic');
        
        if (empty($api_key)) {
            ?>
            <div class="gistmag-blog-generator-notice">
                <p><?php esc_html_e('Please configure your API key in', 'gistmag-blog-generator'); ?> 
                <a href="<?php echo esc_url(admin_url('admin.php?page=gistmag-blog-generator')); ?>">
                    <?php esc_html_e('Settings', 'gistmag-blog-generator'); ?>
                </a></p>
            </div>
            <?php
            return;
        }
        
        // Get post title safely
        $post_title = '';
        if (isset($post) && $post && isset($post->post_title)) {
            $post_title = $post->post_title;
        }
        
        // Add nonce for security
        wp_nonce_field('gistmag_blog_generator_meta', 'gistmag_blog_generator_meta_nonce');
        ?>
        <div id="gistmag-blog-generator-meta-box">
            <p class="description">
                <?php esc_html_e('Generate metadata (images, excerpt, category, and tags) for this post.', 'gistmag-blog-generator'); ?>
            </p>
            
            <div class="gistmag-form-group">
                <label for="gistmag-post-title">
                    <strong><?php esc_html_e('Post Title:', 'gistmag-blog-generator'); ?> <span style="color: red;">*</span></strong>
                </label>
                <input 
                    type="text" 
                    id="gistmag-post-title" 
                    name="gistmag_post_title"
                    value="<?php echo esc_attr($post_title); ?>"
                    placeholder="<?php esc_attr_e('Title will be auto-filled from post if available', 'gistmag-blog-generator'); ?>"
                    class="widefat"
                    style="margin-top: 5px;"
                />
            </div>
            
            <div class="gistmag-form-group" style="margin-top: 15px;">
                <label for="gistmag-image-style">
                    <strong><?php esc_html_e('Image Style:', 'gistmag-blog-generator'); ?></strong>
                </label>
                <select id="gistmag-image-style" name="gistmag_image_style" class="widefat" style="margin-top: 5px;">
                    <option value="realistic" <?php selected($image_style, 'realistic'); ?>><?php esc_html_e('Realistic', 'gistmag-blog-generator'); ?></option>
                    <option value="illustration" <?php selected($image_style, 'illustration'); ?>><?php esc_html_e('Illustration', 'gistmag-blog-generator'); ?></option>
                    <option value="abstract" <?php selected($image_style, 'abstract'); ?>><?php esc_html_e('Abstract', 'gistmag-blog-generator'); ?></option>
                    <option value="minimalist" <?php selected($image_style, 'minimalist'); ?>><?php esc_html_e('Minimalist', 'gistmag-blog-generator'); ?></option>
                    <option value="photographic" <?php selected($image_style, 'photographic'); ?>><?php esc_html_e('Photographic', 'gistmag-blog-generator'); ?></option>
                </select>
            </div>
            
            <button 
                type="button" 
                id="gistmag-generate-btn" 
                class="button button-primary button-large"
                style="width: 100%; margin-top: 15px;"
            >
                <?php esc_html_e('Generate Meta Elements', 'gistmag-blog-generator'); ?>
            </button>
            
            <div id="gistmag-response" style="margin-top: 15px;"></div>
        </div>
        
        <script type="text/javascript">
        // Define postId globally so all functions can access it
        var gistmagPostId = <?php echo intval($post->ID); ?>;
        
        jQuery(document).ready(function($) {
            // Auto-fill title from post editor
            function syncTitle() {
                var editorTitle = '';
                
                // Try classic editor
                if ($('#title').length && $('#title').val()) {
                    editorTitle = $('#title').val().trim();
                }
                
                // Try block editor (Gutenberg)
                if (!editorTitle) {
                    var $h1Title = $('.editor-post-title__input, h1.editor-post-title__input, .wp-block-post-title, h1.wp-block-post-title');
                    if ($h1Title.length) {
                        editorTitle = $h1Title.text().trim() || $h1Title.val() || '';
                    }
                }
                
                // Try block editor via WordPress data API
                if (!editorTitle && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                    try {
                        var editedTitle = wp.data.select('core/editor').getEditedPostAttribute('title');
                        if (editedTitle && editedTitle.trim()) {
                            editorTitle = editedTitle.trim();
                        }
                    } catch (e) {
                        // Block editor not available
                    }
                }
                
                // Update our title field if we found a title and our field is empty
                if (editorTitle && !$('#gistmag-post-title').val().trim()) {
                    $('#gistmag-post-title').val(editorTitle);
                }
            }
            
            // Sync on load
            syncTitle();
            
            // Update when post title changes (classic editor)
            $('#title').on('input change', function() {
                $('#gistmag-post-title').val($(this).val());
            });
            
            // Update when block editor title changes
            $(document).on('input change', '.editor-post-title__input, h1.editor-post-title__input', function() {
                var title = $(this).text().trim() || $(this).val() || '';
                if (title) {
                    $('#gistmag-post-title').val(title);
                }
            });
            
            // Listen for block editor changes via WordPress data API
            if (typeof wp !== 'undefined' && wp.data && wp.data.subscribe) {
                wp.data.subscribe(function() {
                    setTimeout(syncTitle, 100);
                });
            }
            
            $('#gistmag-generate-btn').on('click', function(e) {
                e.preventDefault();
                
                var postTitle = $('#gistmag-post-title').val().trim();
                var imageStyle = $('#gistmag-image-style').val();
                var nonce = $('#gistmag_blog_generator_meta_nonce').val();
                
                // Validation
                if (!postTitle) {
                    $('#gistmag-response').html('<div class="notice notice-error"><p><?php esc_html_e('Please enter a post title first.', 'gistmag-blog-generator'); ?></p></div>');
                    $('#gistmag-post-title').focus().css('border-color', '#d63638');
                    setTimeout(function() {
                        $('#gistmag-post-title').css('border-color', '');
                    }, 2000);
                    return;
                }
                
                // Disable button and show loading
                var $btn = $(this);
                $btn.prop('disabled', true).text('<?php esc_html_e('Generating...', 'gistmag-blog-generator'); ?>');
                $('#gistmag-response').html('<div class="notice notice-info"><p><?php esc_html_e('Generating... This may take a minute.', 'gistmag-blog-generator'); ?></p></div>');
                
                // AJAX request
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'gistmag_blog_generator_generate',
                        post_id: gistmagPostId,
                        title: postTitle,
                        image_style: imageStyle,
                        nonce: nonce
                    },
                    success: function(response) {
                        $btn.prop('disabled', false).text('<?php esc_html_e('Generate Meta Elements', 'gistmag-blog-generator'); ?>');
                        
                        if (response.success) {
                            displayResults(response.data);
                        } else {
                            $('#gistmag-response').html('<div class="notice notice-error"><p>' + (response.data.message || '<?php esc_html_e('An error occurred', 'gistmag-blog-generator'); ?>') + '</p></div>');
                        }
                    },
                    error: function(xhr, status, error) {
                        $btn.prop('disabled', false).text('<?php esc_html_e('Generate Meta Elements', 'gistmag-blog-generator'); ?>');
                        $('#gistmag-response').html('<div class="notice notice-error"><p><?php esc_html_e('Error:', 'gistmag-blog-generator'); ?> ' + error + '</p></div>');
                    }
                });
            });
            
            function displayResults(data) {
                var html = '<div class="gistmag-blog-generator-results">';
                
                // Display images
                if (data.images && data.images.length > 0) {
                    html += '<h4><?php esc_html_e('Generated Images', 'gistmag-blog-generator'); ?> (' + data.images.length + ')</h4>';
                    html += '<div class="gistmag-blog-generator-images" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 10px; margin-bottom: 15px;">';
                    data.images.forEach(function(image, index) {
                        html += '<div style="border: 1px solid #ddd; padding: 5px; position: relative;">';
                        html += '<img src="' + escapeHtml(image.url) + '" alt="Generated image ' + (index + 1) + '" style="width: 100%; height: auto; max-height: 150px; object-fit: cover;">';
                        html += '<p style="font-size: 11px; margin: 5px 0 0 0; text-transform: capitalize;">' + escapeHtml(image.style) + '</p>';
                        html += '<button type="button" class="button button-small" onclick="setFeaturedImage(\'' + escapeHtml(image.url) + '\', ' + gistmagPostId + ')" style="width: 100%; margin-top: 5px;"><?php esc_html_e('Set as Featured', 'gistmag-blog-generator'); ?></button>';
                        html += '</div>';
                    });
                    html += '</div>';
                }
                
                // Display excerpt
                if (data.excerpt) {
                    html += '<h4><?php esc_html_e('Excerpt', 'gistmag-blog-generator'); ?></h4>';
                    html += '<div style="background: #f5f5f5; padding: 10px; border-radius: 4px; margin-bottom: 15px;">';
                    html += '<p style="margin: 0;">' + escapeHtml(data.excerpt) + '</p>';
                    html += '<button type="button" id="gistmag-apply-excerpt-btn" class="button button-small" data-excerpt="' + escapeHtml(data.excerpt) + '" style="margin-top: 10px;"><?php esc_html_e('Apply to Post', 'gistmag-blog-generator'); ?></button>';
                    html += '</div>';
                }
                
                // Display category
                if (data.category) {
                    html += '<h4><?php esc_html_e('Category', 'gistmag-blog-generator'); ?></h4>';
                    html += '<div style="margin-bottom: 15px;">';
                    html += '<span class="button button-small" style="background: #2271b1; color: white; border-color: #2271b1;">' + escapeHtml(data.category) + '</span>';
                    html += '<button type="button" class="button button-small" onclick="applyCategory(\'' + escapeHtml(data.category) + '\')" style="margin-left: 10px;"><?php esc_html_e('Apply to Post', 'gistmag-blog-generator'); ?></button>';
                    html += '</div>';
                }
                
                // Display tags
                if (data.tags && data.tags.length > 0) {
                    html += '<h4><?php esc_html_e('Tags', 'gistmag-blog-generator'); ?></h4>';
                    html += '<div style="margin-bottom: 15px;">';
                    data.tags.forEach(function(tag) {
                        html += '<span class="button button-small" style="margin-right: 5px; margin-bottom: 5px;">' + escapeHtml(tag) + '</span>';
                    });
                    html += '<br><button type="button" id="gistmag-apply-tags-btn" class="button button-small" onclick="applyTags(' + JSON.stringify(data.tags) + ')" style="margin-top: 10px;"><?php esc_html_e('Apply to Post', 'gistmag-blog-generator'); ?></button>';
                    html += '</div>';
                }
                
                html += '</div>';
                $('#gistmag-response').html(html);
                
                // Bind excerpt button click handler
                $('#gistmag-apply-excerpt-btn').on('click', function() {
                    var excerpt = $(this).data('excerpt');
                    applyExcerpt(excerpt);
                });
            }
            
            function escapeHtml(text) {
                var map = {
                    '&': '&amp;',
                    '<': '&lt;',
                    '>': '&gt;',
                    '"': '&quot;',
                    "'": '&#039;'
                };
                return text.replace(/[&<>"']/g, function(m) { return map[m]; });
            }
            
            // Global functions to apply results
            window.applyExcerpt = function(excerpt) {
                // Try multiple selectors for excerpt field
                var $excerptField = $('#excerpt');
                if (!$excerptField.length) {
                    $excerptField = $('#excerpt-textarea');
                }
                if (!$excerptField.length) {
                    $excerptField = $('textarea[name="excerpt"]');
                }
                if (!$excerptField.length) {
                    $excerptField = $('.wp-editor-area[name="excerpt"]');
                }
                
                if ($excerptField.length) {
                    $excerptField.val(excerpt);
                    // Trigger change event for some editors
                    $excerptField.trigger('change');
                    alert('<?php esc_html_e('Excerpt applied! Don\'t forget to save your post.', 'gistmag-blog-generator'); ?>');
                } else {
                    alert('<?php esc_html_e('Excerpt field not found. Please manually paste the excerpt.', 'gistmag-blog-generator'); ?>');
                }
            };
            
            window.applyCategory = function(category) {
                var nonce = $('#gistmag_blog_generator_meta_nonce').val();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'gistmag_blog_generator_apply_category',
                        post_id: gistmagPostId,
                        category: category,
                        nonce: nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            alert('<?php esc_html_e('Category applied successfully!', 'gistmag-blog-generator'); ?>');
                            // Optionally reload page to show updated category (commented out to prevent refresh)
                            // location.reload();
                        } else {
                            alert('<?php esc_html_e('Error applying category:', 'gistmag-blog-generator'); ?> ' + (response.data.message || 'Unknown error'));
                        }
                    },
                    error: function() {
                        alert('<?php esc_html_e('Error applying category. Please try again.', 'gistmag-blog-generator'); ?>');
                    }
                });
            };
            
            window.applyTags = function(tags) {
                // Ensure tags is an array
                if (typeof tags === 'string') {
                    try {
                        tags = JSON.parse(tags);
                    } catch (e) {
                        tags = tags.split(',').map(function(tag) { return tag.trim(); });
                    }
                }
                
                if (!Array.isArray(tags) || tags.length === 0) {
                    alert('<?php esc_html_e('Invalid tags data.', 'gistmag-blog-generator'); ?>');
                    return;
                }
                
                var nonce = $('#gistmag_blog_generator_meta_nonce').val();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    traditional: true, // Use traditional serialization for arrays
                    data: {
                        action: 'gistmag_blog_generator_apply_tags',
                        post_id: gistmagPostId,
                        tags: tags,
                        nonce: nonce
                    },
                    success: function(response) {
                        console.log('Tags response:', response); // Debug
                        if (response && response.success) {
                            // Change button color to show it's been applied
                            var $btn = $('#gistmag-apply-tags-btn');
                            if ($btn.length) {
                                $btn.css({
                                    'background-color': '#00a32a',
                                    'border-color': '#00a32a',
                                    'color': '#fff'
                                }).text('<?php esc_html_e('✓ Tags Applied', 'gistmag-blog-generator'); ?>').prop('disabled', true);
                            }
                            
                            alert('<?php esc_html_e('Tags applied successfully!', 'gistmag-blog-generator'); ?>');
                            // Optionally reload page to show updated tags (commented out to prevent refresh)
                            // location.reload();
                        } else {
                            var errorMsg = response && response.data && response.data.message ? response.data.message : 'Unknown error';
                            alert('<?php esc_html_e('Error applying tags:', 'gistmag-blog-generator'); ?> ' + errorMsg);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Tags AJAX error:', xhr, status, error); // Debug
                        var errorMessage = '<?php esc_html_e('Error applying tags. Please try again.', 'gistmag-blog-generator'); ?>';
                        if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                            errorMessage = xhr.responseJSON.data.message;
                        } else if (xhr.responseText) {
                            try {
                                var errorData = JSON.parse(xhr.responseText);
                                if (errorData.data && errorData.data.message) {
                                    errorMessage = errorData.data.message;
                                }
                            } catch (e) {
                                // Ignore parse error
                            }
                        }
                        alert(errorMessage);
                    }
                });
            };
            
            window.setFeaturedImage = function(imageUrl, postIdParam) {
                var postId = postIdParam || gistmagPostId;
                var nonce = $('#gistmag_blog_generator_meta_nonce').val();
                
                if (!confirm('<?php esc_html_e('Download and set this image as the featured image?', 'gistmag-blog-generator'); ?>')) {
                    return;
                }
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'gistmag_blog_generator_set_featured_image',
                        post_id: postId,
                        image_url: imageUrl,
                        nonce: nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            alert('<?php esc_html_e('Featured image set successfully!', 'gistmag-blog-generator'); ?>');
                            // Optionally reload page to show featured image (commented out to prevent refresh)
                            // location.reload();
                        } else {
                            alert('<?php esc_html_e('Error setting featured image:', 'gistmag-blog-generator'); ?> ' + (response.data.message || 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        var errorMessage = '<?php esc_html_e('Error setting featured image. Please try again.', 'gistmag-blog-generator'); ?>';
                        if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                            errorMessage = xhr.responseJSON.data.message;
                        }
                        alert(errorMessage);
                    }
                });
            };
        });
        </script>
        <?php
    }

    public function ajax_generate() {
        // Verify nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
        $image_style = isset($_POST['image_style']) ? sanitize_text_field(wp_unslash($_POST['image_style'])) : 'realistic';

        if (empty($title)) {
            wp_send_json_error(array('message' => __('Title is required', 'gistmag-blog-generator')));
        }

        // Get post content if available
        $content = '';
        if ($post_id > 0) {
            $post = get_post($post_id);
            if ($post) {
                $content = $post->post_content;
            }
        }

        $result = $this->api->generate_blog_elements($title, $content, $image_style);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        // If post_id is provided and auto-apply is enabled, apply the results
        if ($post_id > 0 && get_option('gistmag_blog_generator_auto_apply', false)) {
            $this->apply_results_to_post($post_id, $result);
        }

        wp_send_json_success($result);
    }

    public function ajax_verify_key() {
        check_ajax_referer('gistmag_blog_generator_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';

        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('API key is required', 'gistmag-blog-generator')));
        }

        $result = $this->api->verify_api_key($api_key);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX handler to apply category
     */
    public function ajax_apply_category() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $category_name = isset($_POST['category']) ? sanitize_text_field(wp_unslash($_POST['category'])) : '';

        if (empty($category_name) || $post_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid data', 'gistmag-blog-generator')));
            return;
        }

        $this->apply_category_to_post($post_id, $category_name);
        wp_send_json_success(array('message' => __('Category applied successfully', 'gistmag-blog-generator')));
    }

    /**
     * AJAX handler to apply tags
     */
    public function ajax_apply_tags() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        // Handle tags - can be array or comma-separated string
        $tags = array();
        if (isset($_POST['tags'])) {
            if (is_array($_POST['tags'])) {
                $tags = array_map('sanitize_text_field', wp_unslash($_POST['tags']));
            } else {
                // Handle as comma-separated string
                $tags_string = sanitize_text_field(wp_unslash($_POST['tags']));
                $tags = array_filter(array_map('trim', explode(',', $tags_string)));
            }
        }

        if (empty($tags) || $post_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid data: tags are required', 'gistmag-blog-generator')));
            return;
        }

        // Apply tags to post
        $result = wp_set_post_tags($post_id, $tags, false);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }

        wp_send_json_success(array('message' => __('Tags applied successfully', 'gistmag-blog-generator'), 'tags_count' => count($tags)));
    }

    /**
     * AJAX handler to set featured image
     */
    public function ajax_set_featured_image() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $image_url = isset($_POST['image_url']) ? esc_url_raw(wp_unslash($_POST['image_url'])) : '';

        if (empty($image_url) || $post_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid data', 'gistmag-blog-generator')));
            return;
        }

        $attachment_id = $this->download_and_set_featured_image($post_id, $image_url);

        if (is_wp_error($attachment_id)) {
            wp_send_json_error(array('message' => $attachment_id->get_error_message()));
            return;
        }

        set_post_thumbnail($post_id, $attachment_id);
        wp_send_json_success(array('message' => __('Featured image set successfully', 'gistmag-blog-generator'), 'attachment_id' => $attachment_id));
    }

    /**
     * Download image from URL and add to media library
     *
     * @param int $post_id Post ID
     * @param string $image_url Image URL
     * @return int|WP_Error Attachment ID or error
     */
    private function download_and_set_featured_image($post_id, $image_url) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Add filter to allow image uploads (temporarily)
        add_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10, 2);
        add_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10, 5);

        // Download the image
        $tmp = download_url($image_url);

        if (is_wp_error($tmp)) {
            remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
            remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
            return $tmp;
        }

        // Parse URL to get file extension
        // phpcs:ignore WordPress.WP.AlternativeFunctions.parse_url_parse_url -- wp_parse_url() doesn't support PHP_URL_PATH constant
        $parsed_url = wp_parse_url($image_url);
        $url_path = isset($parsed_url['path']) ? $parsed_url['path'] : '';
        $file_extension = strtolower(pathinfo($url_path, PATHINFO_EXTENSION));
        
        // If no extension found, try to get from URL or default to jpg
        if (empty($file_extension)) {
            // Check if URL contains image format hints
            if (strpos($image_url, '.jpg') !== false || strpos($image_url, '.jpeg') !== false) {
                $file_extension = 'jpg';
            } elseif (strpos($image_url, '.png') !== false) {
                $file_extension = 'png';
            } elseif (strpos($image_url, '.webp') !== false) {
                $file_extension = 'webp';
            } else {
                // Default to jpg for DALL-E images
                $file_extension = 'jpg';
            }
        }

        // Ensure we have a valid image extension
        $allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'webp');
        if (!in_array($file_extension, $allowed_extensions, true)) {
            $file_extension = 'jpg';
        }

        // Generate a filename with proper extension
        $filename = 'gistmag-generated-' . time() . '-' . wp_generate_password(8, false) . '.' . $file_extension;

        $file_array = array(
            'name' => $filename,
            'tmp_name' => $tmp,
        );

        // Do the validation and storage stuff
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Remove filters
        remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
        remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);

        // If error storing permanently, delete temporary file
        if (is_wp_error($attachment_id)) {
            if (file_exists($file_array['tmp_name'])) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- wp_delete_file() doesn't exist, and this is a temporary file from download_url()
                @unlink($file_array['tmp_name']);
            }
            return $attachment_id;
        }

        return $attachment_id;
    }

    /**
     * Allow image MIME types for upload
     *
     * @param array $mimes Existing MIME types
     * @param int $user_id User ID
     * @return array Modified MIME types
     */
    public function allow_image_upload_mimes($mimes, $user_id) {
        // Ensure common image types are allowed
        $mimes['jpg|jpeg|jpe'] = 'image/jpeg';
        $mimes['png'] = 'image/png';
        $mimes['gif'] = 'image/gif';
        $mimes['webp'] = 'image/webp';
        return $mimes;
    }

    /**
     * Fix filetype detection for images
     *
     * @param array $data File data
     * @param string $file Full path to file
     * @param string $filename Filename
     * @param array $mimes MIME types
     * @param string $real_mime Real MIME type
     * @return array Modified file data
     */
    public function fix_image_filetype($data, $file, $filename, $mimes, $real_mime = null) {
        // If WordPress couldn't detect the file type, try to detect it from extension
        if (empty($data['ext']) || empty($data['type'])) {
            $file_extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            
            $image_types = array(
                'jpg' => 'image/jpeg',
                'jpeg' => 'image/jpeg',
                'png' => 'image/png',
                'gif' => 'image/gif',
                'webp' => 'image/webp',
            );

            if (isset($image_types[$file_extension])) {
                $data['ext'] = $file_extension;
                $data['type'] = $image_types[$file_extension];
            }
        }

        return $data;
    }

    /**
     * Apply category to a post
     *
     * @param int $post_id Post ID
     * @param string $category_name Category name
     */
    private function apply_category_to_post($post_id, $category_name) {
        // Try to find existing category by name
        $category = get_term_by('name', $category_name, 'category');
        
        if (!$category) {
            // Try by slug
            $category = get_term_by('slug', sanitize_title($category_name), 'category');
        }

        if ($category) {
            // Category exists, use it
            wp_set_post_categories($post_id, array($category->term_id), false);
        } else {
            // Create new category
            $category_data = wp_insert_category(array(
                'cat_name' => $category_name,
                'taxonomy' => 'category',
            ));

            if (!is_wp_error($category_data) && $category_data) {
                wp_set_post_categories($post_id, array($category_data), false);
            }
        }
    }

    /**
     * Apply generated results to a post
     *
     * @param int $post_id Post ID
     * @param array $result Generated results
     */
    private function apply_results_to_post($post_id, $result) {
        // Set excerpt
        if (!empty($result['excerpt'])) {
            update_post_meta($post_id, '_gistmag_excerpt', $result['excerpt']);
            // Also update post excerpt if it's empty
            $post = get_post($post_id);
            if ($post && empty($post->post_excerpt)) {
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_excerpt' => $result['excerpt'],
                ));
            }
        }

        // Set category
        if (!empty($result['category'])) {
            $this->apply_category_to_post($post_id, $result['category']);
        }

        // Set tags
        if (!empty($result['tags']) && is_array($result['tags'])) {
            wp_set_post_tags($post_id, $result['tags'], false);
        }

        // Store images
        if (!empty($result['images']) && is_array($result['images'])) {
            update_post_meta($post_id, '_gistmag_generated_images', $result['images']);
        }
    }
}
