(function($) {
    'use strict';

    $(document).ready(function() {
        const $generateButton = $('#gistmag_blog_generator_generate');
        const $resultDiv = $('#gistmag_blog_generator_result');
        const $loadingDiv = $('#gistmag_blog_generator_loading');
        const $contentDiv = $('#gistmag_blog_generator_content');

        // Auto-fill title from post editor when meta box loads or title changes
        function autoFillTitle() {
            const $titleInput = $('#gistmag_blog_generator_title');
            if ($titleInput.val()) {
                return; // Already has a value, don't overwrite
            }
            
            let title = '';
            
            // Try classic editor title field
            if ($('#title').length && $('#title').val()) {
                title = $('#title').val().trim();
            }
            
            // Try block editor title (Gutenberg) - check h1 element
            if (!title) {
                const $h1Title = $('.editor-post-title__input, h1.editor-post-title__input, .wp-block-post-title, h1.wp-block-post-title');
                if ($h1Title.length) {
                    title = $h1Title.text().trim() || $h1Title.val() || '';
                }
            }
            
            // Try block editor via WordPress data API
            if (!title && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                try {
                    const editedTitle = wp.data.select('core/editor').getEditedPostAttribute('title');
                    if (editedTitle && editedTitle.trim()) {
                        title = editedTitle.trim();
                    }
                } catch (e) {
                    // Block editor not available
                }
            }
            
            // Try alternative title selectors
            if (!title) {
                const altTitle = $('input[name="post_title"]').val();
                if (altTitle && altTitle.trim()) {
                    title = altTitle.trim();
                }
            }
            
            // Try from localized script (saved post title)
            if (!title && typeof gistmagBlogGenerator !== 'undefined' && gistmagBlogGenerator.currentPostTitle) {
                title = gistmagBlogGenerator.currentPostTitle.trim();
            }
            
            if (title) {
                $titleInput.val(title);
            }
        }
        
        // Auto-fill on page load
        autoFillTitle();
        
        // Auto-fill when title field in editor changes (for both classic and block editor)
        $(document).on('input change', '#title, .editor-post-title__input, h1.editor-post-title__input', function() {
            setTimeout(autoFillTitle, 100);
        });
        
        // Listen for block editor title changes
        if (typeof wp !== 'undefined' && wp.data && wp.data.subscribe) {
            wp.data.subscribe(function() {
                setTimeout(autoFillTitle, 100);
            });
        }

        $generateButton.on('click', function(e) {
            e.preventDefault();

            // Get title from our dedicated input field (most reliable)
            const $titleInput = $('#gistmag_blog_generator_title');
            
            // Check if field exists
            if (!$titleInput.length) {
                alert('Error: Title input field not found. Please refresh the page.');
                return;
            }
            
            // Get and trim title value
            let title = $titleInput.val();
            if (typeof title === 'string') {
                title = title.trim();
            } else {
                title = '';
            }
            
            // If empty, try to auto-fill one more time
            if (!title) {
                autoFillTitle();
                title = $titleInput.val();
                if (typeof title === 'string') {
                    title = title.trim();
                } else {
                    title = '';
                }
            }
            
            // Get content - try multiple selectors
            let content = '';
            if ($('#content').length) {
                content = $('#content').val() || '';
            }
            
            // Try block editor content
            if (!content && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                try {
                    content = wp.data.select('core/editor').getEditedPostContent() || '';
                } catch (e) {
                    // Block editor not available
                }
            }
            
            const imageStyle = $('#gistmag_blog_generator_image_style').val() || 'realistic';
            const postId = $('#post_ID').val() || 0;

            // Validate title
            if (!title || title.length === 0) {
                alert('Please enter a post title in the "Post Title" field above.');
                $titleInput.focus();
                $titleInput.css('border-color', '#d63638');
                setTimeout(function() {
                    $titleInput.css('border-color', '');
                }, 2000);
                return;
            }
            

            // Show loading state
            $resultDiv.show();
            $loadingDiv.show();
            $contentDiv.hide();
            $generateButton.prop('disabled', true).text('Generating...');

            // Make AJAX request
            $.ajax({
                url: gistmagBlogGenerator.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'gistmag_blog_generator_generate',
                    nonce: gistmagBlogGenerator.nonce,
                    post_id: postId,
                    title: title,
                    content: content,
                    image_style: imageStyle
                },
                success: function(response) {
                    $loadingDiv.hide();
                    $generateButton.prop('disabled', false).text('Generate Meta Elements');

                    if (response.success) {
                        displayResults(response.data);
                    } else {
                        $contentDiv.html('<div class="notice notice-error"><p>' + (response.data.message || 'An error occurred') + '</p></div>');
                        $contentDiv.show();
                    }
                },
                error: function(xhr, status, error) {
                    $loadingDiv.hide();
                    $generateButton.prop('disabled', false).text('Generate Meta Elements');
                    $contentDiv.html('<div class="notice notice-error"><p>Error: ' + error + '</p></div>');
                    $contentDiv.show();
                }
            });
        });

        function displayResults(data) {
            let html = '<div class="gistmag-blog-generator-results">';

            // Display images
            if (data.images && data.images.length > 0) {
                html += '<h4>Generated Images (' + data.images.length + ')</h4>';
                html += '<div class="gistmag-blog-generator-images" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 10px; margin-bottom: 15px;">';
                data.images.forEach(function(image, index) {
                    html += '<div style="border: 1px solid #ddd; padding: 5px;">';
                    html += '<img src="' + escapeHtml(image.url) + '" alt="Generated image ' + (index + 1) + '" style="width: 100%; height: auto; max-height: 150px; object-fit: cover;">';
                    html += '<p style="font-size: 11px; margin: 5px 0 0 0; text-transform: capitalize;">' + escapeHtml(image.style) + '</p>';
                    html += '</div>';
                });
                html += '</div>';
            }

            // Display excerpt
            if (data.excerpt) {
                html += '<h4>Excerpt</h4>';
                html += '<div style="background: #f5f5f5; padding: 10px; border-radius: 4px; margin-bottom: 15px;">';
                html += '<p style="margin: 0;">' + escapeHtml(data.excerpt) + '</p>';
                html += '<button type="button" class="button button-small" onclick="applyExcerpt(\'' + escapeHtml(data.excerpt) + '\')" style="margin-top: 10px;">Apply to Post</button>';
                html += '</div>';
            }

            // Display category
            if (data.category) {
                html += '<h4>Category</h4>';
                html += '<div style="margin-bottom: 15px;">';
                html += '<span class="button button-small" style="background: #2271b1; color: white; border-color: #2271b1;">' + escapeHtml(data.category) + '</span>';
                html += '<button type="button" class="button button-small" onclick="applyCategory(\'' + escapeHtml(data.category) + '\')" style="margin-left: 10px;">Apply to Post</button>';
                html += '</div>';
            }

            // Display tags
            if (data.tags && data.tags.length > 0) {
                html += '<h4>Tags</h4>';
                html += '<div style="margin-bottom: 15px;">';
                data.tags.forEach(function(tag) {
                    html += '<span class="button button-small" style="margin-right: 5px; margin-bottom: 5px;">' + escapeHtml(tag) + '</span>';
                });
                html += '<br><button type="button" class="button button-small" onclick="applyTags(' + JSON.stringify(data.tags) + ')" style="margin-top: 10px;">Apply to Post</button>';
                html += '</div>';
            }

            html += '</div>';
            $contentDiv.html(html);
            $contentDiv.show();
        }

        function escapeHtml(text) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }

        // Global functions to apply results
        window.applyExcerpt = function(excerpt) {
            $('#excerpt').val(excerpt);
            alert('Excerpt applied! Don\'t forget to save your post.');
        };

        window.applyCategory = function(category) {
            // This would need to be implemented based on WordPress category structure
            alert('Category: ' + category + '\n\nPlease manually select this category from the Categories meta box.');
        };

        window.applyTags = function(tags) {
            // Use the same AJAX handler as meta-box.js
            // Get post ID from the page
            let postId = 0;
            if (typeof gistmagBlogGenerator !== 'undefined' && gistmagBlogGenerator.postId) {
                postId = gistmagBlogGenerator.postId;
            } else if ($('#post_ID').length) {
                postId = parseInt($('#post_ID').val()) || 0;
            }
            
            if (!postId || postId <= 0) {
                alert('Error: Post ID not found. Please save your post first.');
                return;
            }
            
            // Get nonce
            let nonce = '';
            if (typeof gistmagBlogGenerator !== 'undefined' && gistmagBlogGenerator.nonce) {
                nonce = gistmagBlogGenerator.nonce;
            } else if ($('#gistmag_blog_generator_meta_nonce').length) {
                nonce = $('#gistmag_blog_generator_meta_nonce').val();
            }
            
            if (!nonce) {
                alert('Error: Security token not found. Please refresh the page.');
                return;
            }
            
            // Ensure tags is an array
            if (!Array.isArray(tags)) {
                tags = [tags];
            }
            
            // Remove empty tags
            tags = tags.filter(function(tag) {
                return tag && typeof tag === 'string' && tag.trim().length > 0;
            });
            
            if (tags.length === 0) {
                alert('Error: No valid tags to apply.');
                return;
            }
            
            // Use AJAX to apply tags (same as meta-box.js)
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                traditional: true,
                data: {
                    action: 'gistmag_blog_generator_apply_tags',
                    post_id: postId,
                    tags: tags,
                    nonce: nonce
                },
                success: function(response) {
                    if (response && response.success) {
                        // Update the tags input field
                        const tagsInput = $('#new-tag-post_tag');
                        if (tagsInput.length && response.data && response.data.tags) {
                            const allTags = response.data.tags.join(',');
                            tagsInput.val(allTags);
                        }
                        alert('Tags applied successfully! Don\'t forget to save your post.');
                    } else {
                        const errorMsg = response && response.data && response.data.message 
                            ? response.data.message 
                            : 'Unknown error';
                        alert('Error applying tags: ' + errorMsg);
                    }
                },
                error: function(xhr, status, error) {
                    let errorMessage = 'Error applying tags. Please try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    }
                    alert(errorMessage);
                }
            });
        };
    });
})(jQuery);

