/**
 * GistMag Blog Meta Generator - Meta Box JavaScript
 * Handles the meta box functionality in the post editor
 */
(function($) {
    'use strict';

    // Define postId globally so all functions can access it
    var gistmagPostId = 0;
    var gistmagBlogGenerator = {};

    // Initialize when document is ready
    jQuery(document).ready(function($) {
        // Get post ID from localized script data
        if (typeof gistmag_blog_generator_meta !== 'undefined') {
            gistmagPostId = gistmag_blog_generator_meta.postId || 0;
            gistmagBlogGenerator = gistmag_blog_generator_meta;
        }

        // Auto-fill title from post editor
        function syncTitle() {
            var editorTitle = '';
            
            // Try classic editor
            if ($('#title').length && $('#title').val()) {
                editorTitle = $('#title').val().trim();
            }
            
            // Try block editor (Gutenberg)
            if (!editorTitle) {
                var $h1Title = $('.editor-post-title__input, h1.editor-post-title__input, .wp-block-post-title, h1.wp-block-post-title');
                if ($h1Title.length) {
                    editorTitle = $h1Title.text().trim() || $h1Title.val() || '';
                }
            }
            
            // Try block editor via WordPress data API
            if (!editorTitle && typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                try {
                    var editedTitle = wp.data.select('core/editor').getEditedPostAttribute('title');
                    if (editedTitle && editedTitle.trim()) {
                        editorTitle = editedTitle.trim();
                    }
                } catch (e) {
                    // Block editor not available
                }
            }
            
            // Update our title field if we found a title and our field is empty
            if (editorTitle && !$('#gistmag-post-title').val().trim()) {
                $('#gistmag-post-title').val(editorTitle);
            }
        }
        
        // Sync on load
        syncTitle();
        
        // Update when post title changes (classic editor)
        $('#title').on('input change', function() {
            $('#gistmag-post-title').val($(this).val());
        });
        
        // Update when block editor title changes
        $(document).on('input change', '.editor-post-title__input, h1.editor-post-title__input', function() {
            var title = $(this).text().trim() || $(this).val() || '';
            if (title) {
                $('#gistmag-post-title').val(title);
            }
        });
        
        // Listen for block editor changes via WordPress data API
        if (typeof wp !== 'undefined' && wp.data && wp.data.subscribe) {
            wp.data.subscribe(function() {
                setTimeout(syncTitle, 100);
            });
        }
        
        $('#gistmag-generate-btn').on('click', function(e) {
            e.preventDefault();
            
            var postTitle = $('#gistmag-post-title').val().trim();
            var imageStyle = $('#gistmag-image-style').val();
            var nonce = $('#gistmag_blog_generator_meta_nonce').val();
            
            // Validation
            if (!postTitle) {
                $('#gistmag-response').html('<div class="notice notice-error"><p>' + (gistmagBlogGenerator.strings.pleaseEnterTitle || 'Please enter a post title first.') + '</p></div>');
                $('#gistmag-post-title').focus().css('border-color', '#d63638');
                setTimeout(function() {
                    $('#gistmag-post-title').css('border-color', '');
                }, 2000);
                return;
            }
            
            // Disable button and show loading
            var $btn = $(this);
            $btn.prop('disabled', true).text(gistmagBlogGenerator.strings.generating || 'Generating...');
            $('#gistmag-response').html('<div class="notice notice-info"><p>' + (gistmagBlogGenerator.strings.generatingMessage || 'Generating... This may take a minute.') + '</p></div>');
            
            // AJAX request
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'gistmag_blog_generator_generate',
                    post_id: gistmagPostId,
                    title: postTitle,
                    image_style: imageStyle,
                    nonce: nonce
                },
                success: function(response) {
                    $btn.prop('disabled', false).text(gistmagBlogGenerator.strings.generateMetaElements || 'Generate Meta Elements');
                    
                    if (response.success) {
                        displayResults(response.data);
                    } else {
                        $('#gistmag-response').html('<div class="notice notice-error"><p>' + (response.data.message || (gistmagBlogGenerator.strings.errorOccurred || 'An error occurred')) + '</p></div>');
                    }
                },
                error: function(xhr, status, error) {
                    $btn.prop('disabled', false).text(gistmagBlogGenerator.strings.generateMetaElements || 'Generate Meta Elements');
                    $('#gistmag-response').html('<div class="notice notice-error"><p>' + (gistmagBlogGenerator.strings.error || 'Error:') + ' ' + error + '</p></div>');
                }
            });
        });
        
        function displayResults(data) {
            var html = '<div class="gistmag-blog-generator-results">';
            
                // Display images
                if (data.images && data.images.length > 0) {
                    html += '<h4>' + (gistmagBlogGenerator.strings.generatedImages || 'Generated Images') + ' (' + data.images.length + ')</h4>';
                    html += '<div class="gistmag-blog-generator-images" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 10px; margin-bottom: 15px;">';
                    data.images.forEach(function(image, index) {
                        var imageUrl = image.url || '';
                        // Escape image URL properly for data attribute (HTML entity encoding)
                        var imageUrlEscaped = imageUrl.replace(/&/g, '&amp;').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
                        html += '<div style="border: 1px solid #ddd; padding: 5px; position: relative;">';
                        html += '<img src="' + escapeHtml(imageUrl) + '" alt="Generated image ' + (index + 1) + '" style="width: 100%; height: auto; max-height: 150px; object-fit: cover;">';
                        html += '<p style="font-size: 11px; margin: 5px 0 0 0; text-transform: capitalize;">' + escapeHtml(image.style) + '</p>';
                        html += '<button type="button" class="button button-small gistmag-set-featured-btn" data-image-url="' + imageUrlEscaped + '" data-post-id="' + gistmagPostId + '" style="width: 100%; margin-top: 5px;">' + (gistmagBlogGenerator.strings.setAsFeatured || 'Set as Featured') + '</button>';
                        html += '</div>';
                    });
                    html += '</div>';
                }
            
            // Display excerpt
            if (data.excerpt) {
                html += '<h4>' + (gistmagBlogGenerator.strings.excerpt || 'Excerpt') + '</h4>';
                html += '<div style="background: #f5f5f5; padding: 10px; border-radius: 4px; margin-bottom: 15px;">';
                html += '<p style="margin: 0;">' + escapeHtml(data.excerpt) + '</p>';
                // Escape excerpt properly for data attribute
                var excerptEscaped = (data.excerpt || '').replace(/&/g, '&amp;').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
                html += '<button type="button" id="gistmag-apply-excerpt-btn" class="button button-small" data-excerpt="' + excerptEscaped + '" style="margin-top: 10px;">' + (gistmagBlogGenerator.strings.applyToPost || 'Apply to Post') + '</button>';
                html += '</div>';
            }
            
            // Display category
            if (data.category) {
                html += '<h4>' + (gistmagBlogGenerator.strings.category || 'Category') + '</h4>';
                html += '<div style="margin-bottom: 15px;">';
                html += '<span class="button button-small" style="background: #2271b1; color: white; border-color: #2271b1;">' + escapeHtml(data.category) + '</span>';
                html += '<button type="button" class="button button-small" onclick="gistmagApplyCategory(\'' + escapeHtml(data.category) + '\')" style="margin-left: 10px;">' + (gistmagBlogGenerator.strings.applyToPost || 'Apply to Post') + '</button>';
                html += '</div>';
            }
            
            // Display tags
            if (data.tags && data.tags.length > 0) {
                html += '<h4>' + (gistmagBlogGenerator.strings.tags || 'Tags') + '</h4>';
                html += '<div style="margin-bottom: 15px;">';
                html += '<p style="font-size: 12px; color: #666; margin-bottom: 10px;">Click on a tag to add it to your post:</p>';
                data.tags.forEach(function(tag, index) {
                    // Escape the tag for HTML attribute (double quotes need to be &quot;)
                    var tagForAttr = String(tag)
                        .replace(/&/g, '&amp;')
                        .replace(/</g, '&lt;')
                        .replace(/>/g, '&gt;')
                        .replace(/"/g, '&quot;')
                        .replace(/'/g, '&#039;');
                    // Escape for display text
                    var tagEscaped = escapeHtml(tag);
                    html += '<button type="button" class="button button-small gistmag-tag-btn" data-tag="' + tagForAttr + '" data-tag-index="' + index + '" style="margin-right: 5px; margin-bottom: 5px; cursor: pointer;">' + tagEscaped + '</button>';
                });
                html += '</div>';
            }
            
            html += '</div>';
            $('#gistmag-response').html(html);
            
            // Bind excerpt button click handler
            $(document).off('click', '#gistmag-apply-excerpt-btn').on('click', '#gistmag-apply-excerpt-btn', function() {
                var $btn = $(this);
                var excerpt = $btn.attr('data-excerpt') || $btn.data('excerpt');
                // Unescape the excerpt (decode HTML entities)
                if (excerpt) {
                    excerpt = excerpt.replace(/&quot;/g, '"').replace(/&#39;/g, "'").replace(/&amp;/g, '&');
                }
                gistmagApplyExcerpt(excerpt);
            });
            
            // Bind featured image button click handlers
            $(document).off('click', '.gistmag-set-featured-btn').on('click', '.gistmag-set-featured-btn', function() {
                var $btn = $(this);
                var imageUrl = $btn.attr('data-image-url') || $btn.data('image-url');
                var postId = $btn.attr('data-post-id') || $btn.data('post-id') || gistmagPostId;
                
                // Unescape the URL (decode HTML entities)
                if (imageUrl) {
                    imageUrl = imageUrl.replace(/&quot;/g, '"').replace(/&#39;/g, "'").replace(/&amp;/g, '&');
                    gistmagSetFeaturedImage(imageUrl, postId);
                } else {
                    alert('Error: Image URL not found');
                }
            });
            
            // Bind tags button click handler using data attribute instead of onclick
            // Handle individual tag button clicks
            $(document).off('click', '.gistmag-tag-btn').on('click', '.gistmag-tag-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var $btn = $(this);
                
                // Get the tag - prefer data attribute, but use button text as fallback
                var tag = null;
                var tagAttr = $btn.attr('data-tag');
                
                if (tagAttr) {
                    // Decode HTML entities (reverse of escapeHtml)
                    // Must decode in order: &amp; last to avoid double-decoding
                    tag = tagAttr
                        .replace(/&quot;/g, '"')
                        .replace(/&#039;/g, "'")
                        .replace(/&lt;/g, '<')
                        .replace(/&gt;/g, '>')
                        .replace(/&amp;/g, '&'); // Do &amp; last
                }
                
                // Fallback to button text if data attribute is missing or empty
                if (!tag || tag.trim().length === 0) {
                    tag = $btn.text().trim();
                }
                
                if (!tag || tag.length === 0) {
                    alert('Error: Invalid tag');
                    return;
                }
                
                // Disable button and show loading state
                var originalText = $btn.text().trim();
                $btn.prop('disabled', true).text('Adding...');
                
                // Apply the single tag
                gistmagApplyTags([tag], $btn, originalText);
            });
        }
        
        function escapeHtml(text) {
            if (typeof text !== 'string') {
                text = String(text);
            }
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }
        
        // Global functions to apply results
        window.gistmagApplyExcerpt = function(excerpt) {
            // Try multiple selectors for excerpt field
            var $excerptField = $('#excerpt');
            if (!$excerptField.length) {
                $excerptField = $('#excerpt-textarea');
            }
            if (!$excerptField.length) {
                $excerptField = $('textarea[name="excerpt"]');
            }
            if (!$excerptField.length) {
                $excerptField = $('.wp-editor-area[name="excerpt"]');
            }
            
            if ($excerptField.length) {
                $excerptField.val(excerpt);
                // Trigger change event for some editors
                $excerptField.trigger('change');
                $excerptField.trigger('input');
                var excerptMsg = gistmagBlogGenerator.strings.excerptApplied || 'Excerpt applied! Don\'t forget to save your post.';
                if (gistmagBlogGenerator.strings.saveHint) {
                    excerptMsg += '\n\n' + gistmagBlogGenerator.strings.saveHint;
                }
                alert(excerptMsg);
            } else {
                // Try block editor excerpt
                if (typeof wp !== 'undefined' && wp.data && wp.data.dispatch) {
                    try {
                        wp.data.dispatch('core/editor').editPost({ excerpt: excerpt });
                        var excerptMsg = gistmagBlogGenerator.strings.excerptApplied || 'Excerpt applied! Don\'t forget to save your post.';
                if (gistmagBlogGenerator.strings.saveHint) {
                    excerptMsg += '\n\n' + gistmagBlogGenerator.strings.saveHint;
                }
                alert(excerptMsg);
                        return;
                    } catch (e) {
                        // Block editor not available or failed
                    }
                }
                // Fallback: copy to clipboard if possible
                if (navigator.clipboard && navigator.clipboard.writeText) {
                    navigator.clipboard.writeText(excerpt).then(function() {
                        alert('Excerpt copied to clipboard! Please paste it into the excerpt field manually.');
                    }).catch(function() {
                        alert(gistmagBlogGenerator.strings.excerptFieldNotFound || 'Excerpt field not found. Please manually paste the excerpt.');
                    });
                } else {
                    alert(gistmagBlogGenerator.strings.excerptFieldNotFound || 'Excerpt field not found. Please manually paste the excerpt.');
                }
            }
        };
        
        window.gistmagApplyCategory = function(category) {
            var nonce = $('#gistmag_blog_generator_meta_nonce').val();
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'gistmag_blog_generator_apply_category',
                    post_id: gistmagPostId,
                    category: category,
                    nonce: nonce
                },
                success: function(response) {
                    if (response.success) {
                        var categoryMsg = gistmagBlogGenerator.strings.categoryApplied || 'Category applied successfully!';
                        if (gistmagBlogGenerator.strings.saveHint) {
                            categoryMsg += '\n\n' + gistmagBlogGenerator.strings.saveHint;
                        }
                        alert(categoryMsg);
                    } else {
                        alert((gistmagBlogGenerator.strings.errorApplyingCategory || 'Error applying category:') + ' ' + (response.data.message || 'Unknown error'));
                    }
                },
                error: function() {
                    alert(gistmagBlogGenerator.strings.errorApplyingCategoryTryAgain || 'Error applying category. Please try again.');
                }
            });
        };
        
        window.gistmagApplyTags = function(tags, $clickedBtn, originalText) {
            // Ensure tags is an array
            if (typeof tags === 'string') {
                try {
                    // Try to parse as JSON first
                    tags = JSON.parse(tags);
                } catch (e) {
                    // If not JSON, try as comma-separated string
                    tags = tags.split(',').map(function(tag) { return tag.trim(); }).filter(function(tag) { return tag.length > 0; });
                }
            }
            
            // If still not an array, make it one
            if (!Array.isArray(tags)) {
                tags = [tags];
            }
            
            if (tags.length === 0) {
                alert(gistmagBlogGenerator.strings.invalidTagsData || 'Invalid tags data.');
                return;
            }
            
            if (!gistmagPostId || gistmagPostId <= 0) {
                alert('Invalid post ID');
                return;
            }
            
            var nonce = $('#gistmag_blog_generator_meta_nonce').val();
            
            // Ensure tags is a proper array
            if (!Array.isArray(tags)) {
                tags = [tags];
            }
            
            // Remove empty tags
            tags = tags.filter(function(tag) {
                return tag && typeof tag === 'string' && tag.trim().length > 0;
            });
            
            if (tags.length === 0) {
                alert(gistmagBlogGenerator.strings.invalidTagsData || 'No valid tags to apply.');
                return;
            }
            
            // Ensure tags is always sent as an array
            // For single tag, make sure it's [tag] not just tag
            var tagsToSend = Array.isArray(tags) ? tags : [tags];
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                traditional: true, // Use traditional serialization for arrays (sends as tags[]=tag1&tags[]=tag2)
                data: {
                    action: 'gistmag_blog_generator_apply_tags',
                    post_id: gistmagPostId,
                    tags: tagsToSend, // jQuery will serialize this as tags[]=tag1&tags[]=tag2 with traditional: true
                    nonce: nonce
                },
                success: function(response) {
                    if (response && response.success) {
                        // Change button color to show it's been applied
                        // Use the clicked button if provided, otherwise find by tag
                        var clickedTag = tags && tags.length > 0 ? tags[0] : null;
                        if ($clickedBtn && $clickedBtn.length) {
                            // Use the passed button reference
                            $clickedBtn.css({
                                'background-color': '#00a32a',
                                'border-color': '#00a32a',
                                'color': '#fff'
                            }).text('✓ ' + clickedTag).prop('disabled', true);
                        } else if (clickedTag) {
                            // Fallback: find button by comparing tag text or data attribute
                            $('.gistmag-tag-btn').each(function() {
                                var $tagBtn = $(this);
                                var tagAttr = $tagBtn.attr('data-tag');
                                if (tagAttr) {
                                    // Decode HTML entities for comparison
                                    var decodedTag = tagAttr
                                        .replace(/&quot;/g, '"')
                                        .replace(/&#039;/g, "'")
                                        .replace(/&amp;/g, '&')
                                        .replace(/&lt;/g, '<')
                                        .replace(/&gt;/g, '>');
                                    if (decodedTag === clickedTag || $tagBtn.text().trim() === clickedTag) {
                                        $tagBtn.css({
                                            'background-color': '#00a32a',
                                            'border-color': '#00a32a',
                                            'color': '#fff'
                                        }).text('✓ ' + clickedTag).prop('disabled', true);
                                    }
                                }
                            });
                        }
                        
                        // Update the tags input field in WordPress UI
                        if (response.data && response.data.tags && Array.isArray(response.data.tags)) {
                            var tagsInput = $('#new-tag-post_tag');
                            if (tagsInput.length) {
                                var currentTags = tagsInput.val() || '';
                                // Split current tags and new tags, remove duplicates, then join with comma (no space)
                                var currentTagsArray = currentTags ? currentTags.split(',').map(function(t) { return t.trim(); }).filter(function(t) { return t.length > 0; }) : [];
                                var allTags = currentTagsArray.concat(response.data.tags);
                                // Remove duplicates (case-insensitive)
                                var uniqueTags = [];
                                var seenTags = {};
                                allTags.forEach(function(tag) {
                                    var tagLower = tag.toLowerCase().trim();
                                    if (tagLower && !seenTags[tagLower]) {
                                        seenTags[tagLower] = true;
                                        uniqueTags.push(tag.trim());
                                    }
                                });
                                // Join all tags with comma (no space) as WordPress expects
                                tagsInput.val(uniqueTags.join(','));
                            }
                            
                            // Also try to update the tag checklist
                            response.data.tags.forEach(function(tag) {
                                var tagSlug = tag.toLowerCase().replace(/\s+/g, '-');
                                var $tagCheckbox = $('#post_tag-' + tagSlug);
                                if ($tagCheckbox.length) {
                                    $tagCheckbox.prop('checked', true);
                                }
                            });
                        }
                        
                        // Show success message for the specific tag
                        var tagName = tags && tags.length > 0 ? tags[0] : 'tag';
                        var successMsg = (gistmagBlogGenerator.strings.tagAppliedSuccess || ('Tag "' + tagName + '" applied successfully!'));
                        if (gistmagBlogGenerator.strings.saveHint) {
                            successMsg += '\n\n' + gistmagBlogGenerator.strings.saveHint;
                        }
                        alert(successMsg);
                    } else {
                        var errorMsg = response && response.data && response.data.message ? response.data.message : 'Unknown error';
                        alert((gistmagBlogGenerator.strings.errorApplyingTags || 'Error applying tags:') + ' ' + errorMsg);
                    }
                },
                error: function(xhr, status, error) {
                    var errorMessage = gistmagBlogGenerator.strings.errorApplyingTagsTryAgain || 'Error applying tag. Please try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (xhr.responseText) {
                        try {
                            var errorData = JSON.parse(xhr.responseText);
                            if (errorData.data && errorData.data.message) {
                                errorMessage = errorData.data.message;
                            }
                        } catch (e) {
                            // Ignore parse error
                        }
                    }
                    alert(errorMessage);
                    // Re-enable the button on error
                    if ($clickedBtn && $clickedBtn.length && originalText) {
                        $clickedBtn.prop('disabled', false).text(originalText).css({
                            'background-color': '',
                            'border-color': '',
                            'color': ''
                        });
                    } else {
                        // Fallback: find and re-enable buttons showing "Adding..."
                        $('.gistmag-tag-btn:disabled').each(function() {
                            var $btn = $(this);
                            if ($btn.text().indexOf('Adding...') !== -1) {
                                var tagAttr = $btn.attr('data-tag');
                                if (tagAttr) {
                                    // Decode HTML entities
                                    var decodedTag = tagAttr
                                        .replace(/&quot;/g, '"')
                                        .replace(/&#039;/g, "'")
                                        .replace(/&amp;/g, '&')
                                        .replace(/&lt;/g, '<')
                                        .replace(/&gt;/g, '>');
                                    $btn.prop('disabled', false).text(decodedTag).css({
                                        'background-color': '',
                                        'border-color': '',
                                        'color': ''
                                    });
                                }
                            }
                        });
                    }
                }
            });
        };
        
        window.gistmagSetFeaturedImage = function(imageUrl, postIdParam) {
            var postId = postIdParam || gistmagPostId;
            var nonce = $('#gistmag_blog_generator_meta_nonce').val();
            
            // Validate inputs
            if (!imageUrl || typeof imageUrl !== 'string' || imageUrl.trim() === '') {
                alert('Invalid data: Image URL is required');
                return;
            }
            
            if (!postId || postId <= 0) {
                alert('Invalid data: Post ID is required');
                return;
            }
            
            if (!nonce) {
                alert('Security error: Nonce is missing');
                return;
            }
            
            // Clean up the image URL
            imageUrl = imageUrl.trim();
            
            if (!confirm(gistmagBlogGenerator.strings.confirmSetFeaturedImage || 'Download and set this image as the featured image?')) {
                return;
            }
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'gistmag_blog_generator_set_featured_image',
                    post_id: postId,
                    image_url: imageUrl,
                    nonce: nonce
                },
                success: function(response) {
                    if (response.success) {
                        var imageMsg = gistmagBlogGenerator.strings.featuredImageSetSuccess || 'Featured image set successfully!';
                        if (gistmagBlogGenerator.strings.saveHint) {
                            imageMsg += '\n\n' + gistmagBlogGenerator.strings.saveHint;
                        }
                        alert(imageMsg);
                        // Optionally refresh the featured image display
                        if (typeof wp !== 'undefined' && wp.media && wp.media.featuredImage) {
                            wp.media.featuredImage.set(response.data.attachment_id);
                        }
                    } else {
                        var errorMsg = response.data && response.data.message 
                            ? response.data.message 
                            : 'Unknown error';
                        alert((gistmagBlogGenerator.strings.errorSettingFeaturedImage || 'Error setting featured image:') + ' ' + errorMsg);
                    }
                },
                error: function(xhr, status, error) {
                    var errorMessage = gistmagBlogGenerator.strings.errorSettingFeaturedImageTryAgain || 'Error setting featured image. Please try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (xhr.responseText) {
                        try {
                            var errorData = JSON.parse(xhr.responseText);
                            if (errorData.data && errorData.data.message) {
                                errorMessage = errorData.data.message;
                            }
                        } catch (e) {
                            // Ignore parse error
                        }
                    }
                    alert(errorMessage);
                }
            });
        };
    });
})(jQuery);


