<?php
/**
 * Main plugin class for GistMag Blog Meta Generator
 *
 * @package GistMag_Blog_Generator
 */

if (!defined('ABSPATH')) {
    exit;
}

class GistMag_Blog_Generator {
    private $api;

    public function __construct() {
        $this->api = new GistMag_Blog_Generator_API();
    }

    public function run() {
        // Enqueue styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_styles'));
        
        // Add meta box to post editor
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        
        // Add AJAX handlers
        add_action('wp_ajax_gistmag_blog_generator_generate', array($this, 'ajax_generate'));
        add_action('wp_ajax_gistmag_blog_generator_verify_key', array($this, 'ajax_verify_key'));
        add_action('wp_ajax_gistmag_blog_generator_apply_category', array($this, 'ajax_apply_category'));
        add_action('wp_ajax_gistmag_blog_generator_apply_tags', array($this, 'ajax_apply_tags'));
        add_action('wp_ajax_gistmag_blog_generator_set_featured_image', array($this, 'ajax_set_featured_image'));
    }

    public function enqueue_styles($hook) {
        // Only load on post edit pages
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }

        wp_enqueue_style(
            'gistmag-blog-generator',
            GISTMAG_BLOG_GENERATOR_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            GISTMAG_BLOG_GENERATOR_VERSION
        );

        // Enqueue JavaScript
        $script_url = GISTMAG_BLOG_GENERATOR_PLUGIN_URL . 'assets/js/meta-box.js';
        wp_enqueue_script(
            'gistmag-blog-generator-meta-box',
            $script_url,
            array('jquery'),
            GISTMAG_BLOG_GENERATOR_VERSION,
            true
        );

        // Localize script with post ID and translated strings
        global $post;
        $post_id = 0;
        if (isset($post) && $post && isset($post->ID)) {
            $post_id = intval($post->ID);
        }

        wp_localize_script('gistmag-blog-generator-meta-box', 'gistmag_blog_generator_meta', array(
            'postId' => $post_id,
            'strings' => array(
                'pleaseEnterTitle' => __('Please enter a post title first.', 'gistmag-blog-generator'),
                'generating' => __('Generating...', 'gistmag-blog-generator'),
                'generatingMessage' => __('Generating... This may take a minute.', 'gistmag-blog-generator'),
                'generateMetaElements' => __('Generate Meta Elements', 'gistmag-blog-generator'),
                'errorOccurred' => __('An error occurred', 'gistmag-blog-generator'),
                'error' => __('Error:', 'gistmag-blog-generator'),
                'generatedImages' => __('Generated Images', 'gistmag-blog-generator'),
                'setAsFeatured' => __('Set as Featured', 'gistmag-blog-generator'),
                'excerpt' => __('Excerpt', 'gistmag-blog-generator'),
                'applyToPost' => __('Apply to Post', 'gistmag-blog-generator'),
                'category' => __('Category', 'gistmag-blog-generator'),
                'tags' => __('Tags', 'gistmag-blog-generator'),
                'excerptApplied' => __('Excerpt applied! Don\'t forget to save your post.', 'gistmag-blog-generator'),
                'excerptFieldNotFound' => __('Excerpt field not found. Please manually paste the excerpt.', 'gistmag-blog-generator'),
                'categoryApplied' => __('Category applied successfully!', 'gistmag-blog-generator'),
                'errorApplyingCategory' => __('Error applying category:', 'gistmag-blog-generator'),
                'errorApplyingCategoryTryAgain' => __('Error applying category. Please try again.', 'gistmag-blog-generator'),
                'invalidTagsData' => __('Invalid tags data.', 'gistmag-blog-generator'),
                'tagsApplied' => __('✓ Tags Applied', 'gistmag-blog-generator'),
                'tagsAppliedSuccess' => __('Tags applied successfully!', 'gistmag-blog-generator'),
                'tagAppliedSuccess' => __('Tag applied successfully!', 'gistmag-blog-generator'),
                'errorApplyingTags' => __('Error applying tags:', 'gistmag-blog-generator'),
                'errorApplyingTagsTryAgain' => __('Error applying tags. Please try again.', 'gistmag-blog-generator'),
                'saveHint' => __('Note: Click "Update" or "Publish" to save all changes to your post.', 'gistmag-blog-generator'),
                'confirmSetFeaturedImage' => __('Download and set this image as the featured image?', 'gistmag-blog-generator'),
                'featuredImageSetSuccess' => __('Featured image set successfully!', 'gistmag-blog-generator'),
                'errorSettingFeaturedImage' => __('Error setting featured image:', 'gistmag-blog-generator'),
                'errorSettingFeaturedImageTryAgain' => __('Error setting featured image. Please try again.', 'gistmag-blog-generator'),
            ),
        ));
    }

    public function add_meta_box() {
        $post_types = get_post_types(array('public' => true));
        foreach ($post_types as $post_type) {
            add_meta_box(
                'gistmag_blog_generator',
                __('GistMag Blog Meta Generator', 'gistmag-blog-generator'),
                array($this, 'render_meta_box'),
                $post_type,
                'side',
                'high'
            );
        }
    }

    public function render_meta_box($post) {
        $api_key = get_option('gistmag_blog_generator_api_key', '');
        $image_style = get_option('gistmag_blog_generator_default_image_style', 'realistic');
        
        if (empty($api_key)) {
            ?>
            <div class="gistmag-blog-generator-notice">
                <p><?php esc_html_e('Please configure your API key in', 'gistmag-blog-generator'); ?> 
                <a href="<?php echo esc_url(admin_url('admin.php?page=gistmag-blog-generator')); ?>">
                    <?php esc_html_e('Settings', 'gistmag-blog-generator'); ?>
                </a></p>
            </div>
            <?php
            return;
        }
        
        // Get post title safely
        $post_title = '';
        if (isset($post) && $post && isset($post->post_title)) {
            $post_title = $post->post_title;
        }
        
        // Add nonce for security
        wp_nonce_field('gistmag_blog_generator_meta', 'gistmag_blog_generator_meta_nonce');
        ?>
        <div id="gistmag-blog-generator-meta-box">
            <p class="description">
                <?php esc_html_e('Generate metadata (images, excerpt, category, and tags) for this post.', 'gistmag-blog-generator'); ?>
            </p>
            <div class="gistmag-save-hint" style="background: #fff3cd; border-left: 4px solid #ffb900; padding: 8px 12px; margin: 10px 0; border-radius: 2px;">
                <p style="margin: 0; font-size: 12px; color: #856404;">
                    <strong><?php esc_html_e('💡 Tip:', 'gistmag-blog-generator'); ?></strong>
                    <?php esc_html_e('Click "Update" or "Publish" to save all changes to your post.', 'gistmag-blog-generator'); ?>
                </p>
            </div>
            
            <div class="gistmag-form-group">
                <label for="gistmag-post-title">
                    <strong><?php esc_html_e('Post Title:', 'gistmag-blog-generator'); ?> <span style="color: red;">*</span></strong>
                </label>
                <input 
                    type="text" 
                    id="gistmag-post-title" 
                    name="gistmag_post_title"
                    value="<?php echo esc_attr($post_title); ?>"
                    placeholder="<?php esc_attr_e('Title will be auto-filled from post if available', 'gistmag-blog-generator'); ?>"
                    class="widefat"
                    style="margin-top: 5px;"
                />
            </div>
            
            <div class="gistmag-form-group" style="margin-top: 15px;">
                <label for="gistmag-image-style">
                    <strong><?php esc_html_e('Image Style:', 'gistmag-blog-generator'); ?></strong>
                </label>
                <select id="gistmag-image-style" name="gistmag_image_style" class="widefat" style="margin-top: 5px;">
                    <option value="realistic" <?php selected($image_style, 'realistic'); ?>><?php esc_html_e('Realistic', 'gistmag-blog-generator'); ?></option>
                    <option value="illustration" <?php selected($image_style, 'illustration'); ?>><?php esc_html_e('Illustration', 'gistmag-blog-generator'); ?></option>
                    <option value="abstract" <?php selected($image_style, 'abstract'); ?>><?php esc_html_e('Abstract', 'gistmag-blog-generator'); ?></option>
                    <option value="minimalist" <?php selected($image_style, 'minimalist'); ?>><?php esc_html_e('Minimalist', 'gistmag-blog-generator'); ?></option>
                    <option value="photographic" <?php selected($image_style, 'photographic'); ?>><?php esc_html_e('Photographic', 'gistmag-blog-generator'); ?></option>
                </select>
            </div>
            
            <button 
                type="button" 
                id="gistmag-generate-btn" 
                class="button button-primary button-large"
                style="width: 100%; margin-top: 15px;"
            >
                <?php esc_html_e('Generate Meta Elements', 'gistmag-blog-generator'); ?>
            </button>
            
            <div id="gistmag-response" style="margin-top: 15px;"></div>
        </div>
        <?php
    }

    public function ajax_generate() {
        // Verify nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
        $image_style = isset($_POST['image_style']) ? sanitize_text_field(wp_unslash($_POST['image_style'])) : 'realistic';

        if (empty($title)) {
            wp_send_json_error(array('message' => __('Title is required', 'gistmag-blog-generator')));
        }

        // Get post content if available
        $content = '';
        if ($post_id > 0) {
            $post = get_post($post_id);
            if ($post) {
                $content = $post->post_content;
            }
        }

        $result = $this->api->generate_blog_elements($title, $content, $image_style);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        // If post_id is provided and auto-apply is enabled, apply the results
        if ($post_id > 0 && get_option('gistmag_blog_generator_auto_apply', false)) {
            $this->apply_results_to_post($post_id, $result);
        }

        wp_send_json_success($result);
    }

    public function ajax_verify_key() {
        check_ajax_referer('gistmag_blog_generator_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
        }

        $api_key = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';

        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('API key is required', 'gistmag-blog-generator')));
        }

        $result = $this->api->verify_api_key($api_key);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX handler to apply category
     */
    public function ajax_apply_category() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $category_name = isset($_POST['category']) ? sanitize_text_field(wp_unslash($_POST['category'])) : '';

        if (empty($category_name) || $post_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid data', 'gistmag-blog-generator')));
            return;
        }

        $category_id = $this->apply_category_to_post($post_id, $category_name);
        
        if (is_wp_error($category_id)) {
            wp_send_json_error(array('message' => $category_id->get_error_message()));
            return;
        }

        // Also update the post to ensure category is saved
        wp_update_post(array(
            'ID' => $post_id,
        ));

        wp_send_json_success(array(
            'message' => __('Category applied successfully', 'gistmag-blog-generator'),
            'category_id' => $category_id,
            'category_name' => $category_name
        ));
    }

    /**
     * AJAX handler to apply tags
     */
    public function ajax_apply_tags() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        // Handle tags - jQuery with traditional:true sends as tags[]=tag1&tags[]=tag2
        // Check for both $_POST['tags'] (array) and $_POST['tags[]'] (if jQuery sends it differently)
        $tags = array();
        
        // First, check if tags is already an array (most common case)
        if (isset($_POST['tags']) && is_array($_POST['tags'])) {
            $tags = array_values(array_map('sanitize_text_field', wp_unslash($_POST['tags'])));
        }
        // Check for tags[] format (jQuery traditional serialization)
        elseif (isset($_POST['tags[]']) && is_array($_POST['tags[]'])) {
            $tags = array_values(array_map('sanitize_text_field', wp_unslash($_POST['tags[]'])));
        }
        // Check if tags is a string (comma-separated or JSON)
        elseif (isset($_POST['tags']) && is_string($_POST['tags'])) {
            $tags_string = sanitize_text_field(wp_unslash($_POST['tags']));
            // Try to parse as JSON first
            $tags_json = json_decode(stripslashes($tags_string), true);
            if (is_array($tags_json)) {
                $tags = array_values(array_map('sanitize_text_field', wp_unslash($tags_json)));
            } else {
                // Handle as comma-separated string
                $tags = array_values(array_filter(array_map('trim', explode(',', $tags_string))));
            }
        }

        if (empty($tags) || $post_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid data: tags are required', 'gistmag-blog-generator')));
            return;
        }
        
        // Remove empty tags and ensure all are strings
        $tags = array_values(array_filter(array_map('trim', $tags), function($tag) {
            return !empty($tag) && is_string($tag);
        }));

        if (empty($tags)) {
            wp_send_json_error(array('message' => __('Invalid data: no valid tags provided', 'gistmag-blog-generator')));
            return;
        }

        // For individual tag additions, use append mode to avoid race conditions
        // This ensures each tag is added without overwriting others
        $result = wp_set_post_tags($post_id, $tags, true);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }

        // Get the applied tags to return in response
        $applied_tags = wp_get_post_tags($post_id, array('fields' => 'names'));

        // Also update the post to ensure tags are saved
        wp_update_post(array(
            'ID' => $post_id,
        ));

        wp_send_json_success(array(
            'message' => __('Tags applied successfully', 'gistmag-blog-generator'),
            'tags' => $applied_tags
        ));
    }

    /**
     * AJAX handler to set featured image
     */
    public function ajax_set_featured_image() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_blog_generator_meta')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'gistmag-blog-generator')));
            return;
        }

        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'gistmag-blog-generator')));
            return;
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $image_url = isset($_POST['image_url']) ? esc_url_raw(wp_unslash($_POST['image_url'])) : '';

        // Debug logging (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            error_log('GistMag Blog Generator - Featured Image Request: post_id=' . $post_id . ', image_url=' . $image_url);
        }

        if (empty($image_url)) {
            wp_send_json_error(array('message' => __('Invalid data: image URL is required', 'gistmag-blog-generator')));
            return;
        }

        if ($post_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid data: post ID is required', 'gistmag-blog-generator')));
            return;
        }

        // Verify post exists
        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(array('message' => __('Post not found', 'gistmag-blog-generator')));
            return;
        }

        $attachment_id = $this->download_and_set_featured_image($post_id, $image_url);

        if (is_wp_error($attachment_id)) {
            wp_send_json_error(array('message' => $attachment_id->get_error_message()));
            return;
        }

        if (!$attachment_id || $attachment_id <= 0) {
            wp_send_json_error(array('message' => __('Failed to create attachment', 'gistmag-blog-generator')));
            return;
        }

        $result = set_post_thumbnail($post_id, $attachment_id);
        
        if (!$result) {
            wp_send_json_error(array('message' => __('Failed to set featured image', 'gistmag-blog-generator')));
            return;
        }

        wp_send_json_success(array('message' => __('Featured image set successfully', 'gistmag-blog-generator'), 'attachment_id' => $attachment_id));
    }

    /**
     * Download image from URL and add to media library
     *
     * @param int $post_id Post ID
     * @param string $image_url Image URL (regular URL from Supabase Storage, or data URL as fallback)
     * @return int|WP_Error Attachment ID or error
     */
    private function download_and_set_featured_image($post_id, $image_url) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Add filter to allow image uploads (temporarily)
        add_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10, 2);
        add_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10, 5);

        // Handle data URLs (base64 images - fallback if storage upload fails)
        // Note: Images are now primarily returned as regular URLs from Supabase Storage
        $tmp = false;
        if (strpos($image_url, 'data:image/') === 0) {
            // Extract base64 data and MIME type
            if (preg_match('/^data:image\/(\w+);base64,(.+)$/', $image_url, $matches)) {
                $mime_type = $matches[1];
                $base64_data = $matches[2];
                
                // Decode base64 data
                $image_data = base64_decode($base64_data, true);
                if ($image_data === false) {
                    remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                    remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                    return new WP_Error('invalid_base64', __('Invalid base64 image data', 'gistmag-blog-generator'));
                }
                
                // Create temporary file with proper extension
                $file_extension = ($mime_type === 'jpeg' || $mime_type === 'jpg') ? 'jpg' : $mime_type;
                $tmp = wp_tempnam('gistmag-image-' . time() . '-');
                if ($tmp === false) {
                    remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                    remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                    return new WP_Error('temp_file_failed', __('Failed to create temporary file', 'gistmag-blog-generator'));
                }
                
                // Rename temp file to have proper extension using WP_Filesystem
                $tmp_with_ext = $tmp . '.' . $file_extension;
                global $wp_filesystem;
                if (!function_exists('WP_Filesystem')) {
                    require_once ABSPATH . 'wp-admin/includes/file.php';
                }
                WP_Filesystem();
                if ($wp_filesystem && $wp_filesystem->exists($tmp)) {
                    if (!$wp_filesystem->move($tmp, $tmp_with_ext, true)) {
                        wp_delete_file($tmp);
                        remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                        remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                        return new WP_Error('rename_failed', __('Failed to rename temporary file', 'gistmag-blog-generator'));
                    }
                } else {
                    // Fallback to rename if WP_Filesystem is not available
                    if (!@rename($tmp, $tmp_with_ext)) {
                        wp_delete_file($tmp);
                        remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                        remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                        return new WP_Error('rename_failed', __('Failed to rename temporary file', 'gistmag-blog-generator'));
                    }
                }
                $tmp = $tmp_with_ext;
                
                // Write image data to temporary file
                $written = file_put_contents($tmp, $image_data);
                if ($written === false) {
                    wp_delete_file($tmp);
                    remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                    remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                    return new WP_Error('write_failed', __('Failed to write image data to temporary file', 'gistmag-blog-generator'));
                }
            } else {
                remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                return new WP_Error('invalid_data_url', __('Invalid data URL format', 'gistmag-blog-generator'));
            }
        } else {
            // Regular URL - download it
            $tmp = download_url($image_url);

            if (is_wp_error($tmp)) {
                remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
                remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);
                return $tmp;
            }

            // Parse URL to get file extension
            // phpcs:ignore WordPress.WP.AlternativeFunctions.parse_url_parse_url -- wp_parse_url() doesn't support PHP_URL_PATH constant
            $parsed_url = wp_parse_url($image_url);
            $url_path = isset($parsed_url['path']) ? $parsed_url['path'] : '';
            $file_extension = strtolower(pathinfo($url_path, PATHINFO_EXTENSION));
            
            // If no extension found, try to get from URL or default to jpg
            if (empty($file_extension)) {
                // Check if URL contains image format hints
                if (strpos($image_url, '.jpg') !== false || strpos($image_url, '.jpeg') !== false) {
                    $file_extension = 'jpg';
                } elseif (strpos($image_url, '.png') !== false) {
                    $file_extension = 'png';
                } elseif (strpos($image_url, '.webp') !== false) {
                    $file_extension = 'webp';
                } else {
                    // Default to jpg for DALL-E images
                    $file_extension = 'jpg';
                }
            }
        }

        // Ensure we have a valid image extension
        $allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'webp');
        if (!in_array($file_extension, $allowed_extensions, true)) {
            $file_extension = 'jpg';
        }

        // Generate a filename with proper extension
        $filename = 'gistmag-generated-' . time() . '-' . wp_generate_password(8, false) . '.' . $file_extension;

        $file_array = array(
            'name' => $filename,
            'tmp_name' => $tmp,
        );

        // Do the validation and storage stuff
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Remove filters
        remove_filter('upload_mimes', array($this, 'allow_image_upload_mimes'), 10);
        remove_filter('wp_check_filetype_and_ext', array($this, 'fix_image_filetype'), 10);

        // If error storing permanently, delete temporary file
        if (is_wp_error($attachment_id)) {
            if (file_exists($file_array['tmp_name'])) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- wp_delete_file() doesn't exist, and this is a temporary file from download_url()
                @unlink($file_array['tmp_name']);
            }
            return $attachment_id;
        }

        // Verify attachment was created
        if (!$attachment_id || $attachment_id <= 0) {
            if (file_exists($file_array['tmp_name'])) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
                @unlink($file_array['tmp_name']);
            }
            return new WP_Error('attachment_failed', __('Failed to create attachment', 'gistmag-blog-generator'));
        }

        return $attachment_id;
    }

    /**
     * Allow image MIME types for upload
     *
     * @param array $mimes Existing MIME types
     * @param int $user_id User ID
     * @return array Modified MIME types
     */
    public function allow_image_upload_mimes($mimes, $user_id) {
        // Ensure common image types are allowed
        $mimes['jpg|jpeg|jpe'] = 'image/jpeg';
        $mimes['png'] = 'image/png';
        $mimes['gif'] = 'image/gif';
        $mimes['webp'] = 'image/webp';
        return $mimes;
    }

    /**
     * Fix filetype detection for images
     *
     * @param array $data File data
     * @param string $file Full path to file
     * @param string $filename Filename
     * @param array $mimes MIME types
     * @param string $real_mime Real MIME type
     * @return array Modified file data
     */
    public function fix_image_filetype($data, $file, $filename, $mimes, $real_mime = null) {
        // If WordPress couldn't detect the file type, try to detect it from extension
        if (empty($data['ext']) || empty($data['type'])) {
            $file_extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            
            $image_types = array(
                'jpg' => 'image/jpeg',
                'jpeg' => 'image/jpeg',
                'png' => 'image/png',
                'gif' => 'image/gif',
                'webp' => 'image/webp',
            );

            if (isset($image_types[$file_extension])) {
                $data['ext'] = $file_extension;
                $data['type'] = $image_types[$file_extension];
            }
        }

        return $data;
    }

    /**
     * Apply category to a post
     *
     * @param int $post_id Post ID
     * @param string $category_name Category name
     */
    private function apply_category_to_post($post_id, $category_name) {
        // Try to find existing category by name
        $category = get_term_by('name', $category_name, 'category');
        
        if (!$category) {
            // Try by slug
            $category = get_term_by('slug', sanitize_title($category_name), 'category');
        }

        if ($category) {
            // Category exists, use it
            wp_set_post_categories($post_id, array($category->term_id), false);
        } else {
            // Create new category
            $category_data = wp_insert_category(array(
                'cat_name' => $category_name,
                'taxonomy' => 'category',
            ));

            if (!is_wp_error($category_data) && $category_data) {
                wp_set_post_categories($post_id, array($category_data), false);
            }
        }
    }

    /**
     * Apply generated results to a post
     *
     * @param int $post_id Post ID
     * @param array $result Generated results
     */
    private function apply_results_to_post($post_id, $result) {
        // Set excerpt
        if (!empty($result['excerpt'])) {
            update_post_meta($post_id, '_gistmag_excerpt', $result['excerpt']);
            // Also update post excerpt if it's empty
            $post = get_post($post_id);
            if ($post && empty($post->post_excerpt)) {
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_excerpt' => $result['excerpt'],
                ));
            }
        }

        // Set category
        if (!empty($result['category'])) {
            $this->apply_category_to_post($post_id, $result['category']);
        }

        // Set tags
        if (!empty($result['tags']) && is_array($result['tags'])) {
            wp_set_post_tags($post_id, $result['tags'], false);
        }

        // Store images
        if (!empty($result['images']) && is_array($result['images'])) {
            update_post_meta($post_id, '_gistmag_generated_images', $result['images']);
        }
    }
}
