/**
 * GistMag TTS Reader - Admin Settings JavaScript
 * Handles voice selection and dynamic loading
 */
(function($) {
    'use strict';

    var loadingVoices = false;
    var currentVoices = [];

    $(document).ready(function() {
        var $voiceSelect = $('#gistmag_tts_reader_voice_name');
        var $languageSelect = $('#gistmag_tts_reader_language');
        var $loadingSpan = $('#gistmag-tts-voice-loading');

        // Load voices on page load
        if (typeof gistmagTTSReaderAdmin !== 'undefined') {
            loadVoices(gistmagTTSReaderAdmin.currentLanguage, gistmagTTSReaderAdmin.currentVoice);
        }

        // Load voices when language changes
        $languageSelect.on('change', function() {
            var selectedLanguage = $(this).val();
            loadVoices(selectedLanguage, '');
        });

        /**
         * Load voices from API
         */
        function loadVoices(language, selectedVoice) {
            if (loadingVoices) {
                return;
            }

            loadingVoices = true;
            $loadingSpan.show();
            $voiceSelect.prop('disabled', true);

            // Clear existing options except the first one
            $voiceSelect.find('option:not(:first)').remove();

            $.ajax({
                url: gistmagTTSReaderAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'gistmag_tts_get_voices',
                    language: language || 'en',
                    nonce: gistmagTTSReaderAdmin.nonce
                },
                success: function(response) {
                    if (response.success && response.data && response.data.voices) {
                        currentVoices = response.data.voices;
                        
                        // Sort voices by friendly name
                        currentVoices.sort(function(a, b) {
                            var nameA = (a.friendly_name || a.name || '').toLowerCase();
                            var nameB = (b.friendly_name || b.name || '').toLowerCase();
                            return nameA.localeCompare(nameB);
                        });

                        // Add voice options
                        currentVoices.forEach(function(voice) {
                            var displayName = voice.friendly_name || voice.name;
                            var optionValue = voice.name;
                            
                            // Add additional info if available
                            if (voice.model) {
                                displayName += ' (' + voice.model + ')';
                            }
                            if (voice.ssml_gender) {
                                displayName += ' - ' + voice.ssml_gender;
                            }

                            var $option = $('<option></option>')
                                .attr('value', optionValue)
                                .text(displayName);

                            // Select if this is the current voice
                            if (selectedVoice && optionValue === selectedVoice) {
                                $option.prop('selected', true);
                            }

                            $voiceSelect.append($option);
                        });

                        // If no voice was selected and we have voices, try to match by name
                        if (selectedVoice && !$voiceSelect.val()) {
                            // Try to find a matching voice
                            var matchingVoice = currentVoices.find(function(v) {
                                return v.name === selectedVoice || v.name.indexOf(selectedVoice) !== -1;
                            });
                            if (matchingVoice) {
                                $voiceSelect.val(matchingVoice.name);
                            }
                        }
                    } else {
                        // Show error message
                        var errorMsg = response.data && response.data.message 
                            ? response.data.message 
                            : 'Failed to load voices';
                        $voiceSelect.append($('<option></option>').text('Error: ' + errorMsg).attr('disabled', true));
                    }
                },
                error: function(xhr, status, error) {
                    $voiceSelect.append($('<option></option>').text('Error loading voices').attr('disabled', true));
                },
                complete: function() {
                    loadingVoices = false;
                    $loadingSpan.hide();
                    $voiceSelect.prop('disabled', false);
                }
            });
        }
    });
})(jQuery);

