(function($) {
    'use strict';

    class GistMagTTSPlayer {
        constructor(element) {
            this.element = $(element);
            this.postId = this.element.data('post-id');
            this.playerType = this.element.data('player-type') || 'simple';
            this.audio = null;
            this.isPlaying = false;
            this.isLoading = false;
            this.currentStreamUrl = null;
            // Safely get settings from localized script
            if (typeof gistmagTTSReader !== 'undefined') {
                this.currentSpeed = parseFloat(gistmagTTSReader.speed) || 1.0;
                this.currentVoice = gistmagTTSReader.voiceName || '';
            } else {
                this.currentSpeed = 1.0;
                this.currentVoice = '';
            }
            
            this.init();
        }

        init() {
            this.bindEvents();
            if (this.playerType === 'full') {
                this.initSpeedControl();
                // Ensure settings button is clickable
                this.ensureSettingsButtonClickable();
            }
        }

        ensureSettingsButtonClickable() {
            const self = this;
            const settingsBtn = this.element.find('.gistmag-tts-settings-toggle');
            if (settingsBtn.length > 0) {
                // Remove any existing handlers and re-bind
                settingsBtn.off('click').on('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                    self.toggleSettings();
                    return false;
                });
            }
        }

        bindEvents() {
            const self = this;
            
            // Play/Pause button
            this.element.find('.gistmag-tts-play-btn').on('click', function() {
                if (self.isPlaying) {
                    self.pause();
                } else {
                    self.play();
                }
            });

            // Progress bar click
            this.element.find('.gistmag-tts-progress-bar').on('click', function(e) {
                if (self.audio && self.audio.duration) {
                    const rect = this.getBoundingClientRect();
                    const clickX = e.clientX - rect.left;
                    const percentage = clickX / rect.width;
                    const newTime = percentage * self.audio.duration;
                    self.audio.currentTime = newTime;
                }
            });

            // Settings toggle (full player only)
            if (this.playerType === 'full') {
                this.element.find('.gistmag-tts-settings-toggle').on('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.toggleSettings();
                    return false;
                });
            }
        }

        initSpeedControl() {
            const self = this;
            const speedSlider = this.element.find('.gistmag-tts-speed-slider');
            const speedValue = this.element.find('.gistmag-tts-speed-value');
            
            // Set initial value
            speedSlider.val(this.currentSpeed);
            speedValue.text(this.currentSpeed.toFixed(2) + 'x');
            
            // Handle speed change
            speedSlider.on('input', function() {
                const newSpeed = parseFloat($(this).val());
                self.currentSpeed = newSpeed;
                speedValue.text(newSpeed.toFixed(2) + 'x');
                
                // If audio is already loaded, use playbackRate (no restart needed)
                if (self.audio && self.audio.readyState >= 2) {
                    // Audio is loaded, just change playback rate
                    if (self.audio.playbackRate !== undefined) {
                        self.audio.playbackRate = newSpeed;
                    } else {
                        // playbackRate not supported, need to restart
                        self.restartWithNewSpeed(newSpeed);
                    }
                } else if (self.audio && self.isPlaying) {
                    // Audio is loading/playing but not ready, restart with new speed
                    self.restartWithNewSpeed(newSpeed);
                }
                // If audio doesn't exist yet, speed will be used when it starts
            });
        }


        toggleSettings() {
            const settingsPanel = this.element.find('.gistmag-tts-settings-panel');
            settingsPanel.slideToggle(200);
        }

        restartWithNewSpeed(newSpeed) {
            const self = this;
            if (!this.audio) {
                return;
            }
            
            const wasPlaying = this.isPlaying;
            const currentTime = this.audio.currentTime;
            
            this.pause();
            this.audio = null;
            this.currentStreamUrl = null;
            
            // Restart with new speed
            setTimeout(function() {
                self.play().then(function() {
                    if (wasPlaying && self.audio && currentTime > 0) {
                        // Try to resume at the same position (approximate)
                        self.audio.addEventListener('loadedmetadata', function() {
                            self.audio.currentTime = Math.min(currentTime, self.audio.duration);
                        }, { once: true });
                    }
                }).catch(function(err) {
                    // Error restarting with new speed
                });
            }, 100);
        }

        async play() {
            if (this.isLoading) {
                return;
            }

            // If speed changed and audio exists, we need to restart
            if (this.audio && this.currentStreamUrl) {
                // Check if speed changed
                const speedSlider = this.element.find('.gistmag-tts-speed-slider');
                let speedChanged = false;
                if (speedSlider.length) {
                    const newSpeed = parseFloat(speedSlider.val());
                    if (Math.abs(newSpeed - this.currentSpeed) > 0.01) {
                        speedChanged = true;
                    }
                }
                
                if (speedChanged) {
                    // Speed changed, need to restart
                    this.audio = null;
                    this.currentStreamUrl = null;
                } else {
                    // Resume existing audio
                        this.audio.play().catch(err => {
                            this.showError('Failed to play audio');
                        });
                    return;
                }
            }

            // Start new stream
            this.isLoading = true;
            this.updateStatus('Loading...');
            this.showError(''); // Clear any previous errors

            try {
                // Get current speed
                const speedSlider = this.element.find('.gistmag-tts-speed-slider');
                if (speedSlider.length) {
                    this.currentSpeed = parseFloat(speedSlider.val());
                }

                // Get post content
                const response = await $.ajax({
                    url: gistmagTTSReader.ajaxUrl,
                    method: 'POST',
                    data: {
                        action: 'gistmag_tts_stream',
                        post_id: this.postId,
                        speed: this.currentSpeed,
                        nonce: gistmagTTSReader.nonce,
                    },
                });

                if (!response.success) {
                    throw new Error(response.data?.message || 'Failed to start TTS stream');
                }

                // Create streaming URL with current speed
                const streamUrl = this.createStreamUrl(response.data.post_id);
                
                // Create audio element
                this.audio = new Audio();
                this.currentStreamUrl = streamUrl;
                
                // Set up audio event listeners first
                this.setupAudioEvents();
                
                // Set playback rate before setting src (if supported)
                if (this.audio.playbackRate !== undefined) {
                    this.audio.playbackRate = this.currentSpeed;
                }
                
                // Start loading and playing
                this.audio.src = streamUrl;
                
                // Set playback rate again after setting src
                if (this.audio.playbackRate !== undefined) {
                    this.audio.playbackRate = this.currentSpeed;
                }
                
                await this.audio.play();
                
            } catch (error) {
                this.showError(error.message || 'Failed to load audio');
                this.updateStatus('Error');
                this.isLoading = false;
            }
        }

        createStreamUrl(postId) {
            // Create a URL that will proxy the stream
            const params = new URLSearchParams({
                action: 'gistmag_tts_proxy',
                post_id: postId,
                speed: this.currentSpeed,
                nonce: gistmagTTSReader.nonce,
            });
            
            // Add voice if selected
            const voiceSelect = this.element.find('.gistmag-tts-voice-select');
            // Use voice from admin settings (if set)
            if (this.currentVoice) {
                params.append('voice', this.currentVoice);
            }
            
            return gistmagTTSReader.ajaxUrl + '?' + params.toString();
        }

        setupAudioEvents() {
            const self = this;

            this.audio.addEventListener('loadstart', function() {
                self.updateStatus('Loading audio...');
            });

            this.audio.addEventListener('loadedmetadata', function() {
                // Always apply playback rate when metadata is loaded
                // This ensures the speed is set correctly
                if (self.audio.playbackRate !== undefined) {
                    self.audio.playbackRate = self.currentSpeed;
                }
                self.updateDuration(self.audio.duration);
                self.isLoading = false;
                self.updatePlayButton();
                self.updateStatus('Ready');
            });
            
            // Also set playbackRate when audio can play
            this.audio.addEventListener('canplay', function() {
                if (self.audio.playbackRate !== undefined) {
                    self.audio.playbackRate = self.currentSpeed;
                }
            });

            this.audio.addEventListener('timeupdate', function() {
                self.updateProgress();
            });

            this.audio.addEventListener('play', function() {
                // Ensure playback rate is set when playing starts
                if (self.audio.playbackRate !== undefined) {
                    self.audio.playbackRate = self.currentSpeed;
                }
                self.isPlaying = true;
                self.updatePlayButton();
                self.updateStatus('Playing');
            });

            this.audio.addEventListener('pause', function() {
                self.isPlaying = false;
                self.updatePlayButton();
                self.updateStatus('Paused');
            });

            this.audio.addEventListener('ended', function() {
                self.isPlaying = false;
                self.updatePlayButton();
                self.updateStatus('Finished');
                self.resetProgress();
            });

            this.audio.addEventListener('error', function(e) {
                let errorMsg = 'Failed to play audio. Please try again.';
                
                // More specific error messages
                if (this.error) {
                    switch(this.error.code) {
                        case MediaError.MEDIA_ERR_ABORTED:
                            errorMsg = 'Audio playback was aborted.';
                            break;
                        case MediaError.MEDIA_ERR_NETWORK:
                            errorMsg = 'Network error. Please check your connection.';
                            break;
                        case MediaError.MEDIA_ERR_DECODE:
                            errorMsg = 'Audio decoding error. The file may be corrupted.';
                            break;
                        case MediaError.MEDIA_ERR_SRC_NOT_SUPPORTED:
                            errorMsg = 'Audio format not supported or source not found.';
                            break;
                    }
                }
                
                self.showError(errorMsg);
                self.updateStatus('Error');
                self.isLoading = false;
                self.isPlaying = false;
                self.updatePlayButton();
            });

            this.audio.addEventListener('waiting', function() {
                self.updateStatus('Buffering...');
            });

            this.audio.addEventListener('canplay', function() {
                self.updateStatus('Ready');
            });
        }

        pause() {
            if (this.audio) {
                this.audio.pause();
            }
        }

        updatePlayButton() {
            const playIcon = this.element.find('.gistmag-tts-play-icon');
            const pauseIcon = this.element.find('.gistmag-tts-pause-icon');
            
            if (this.isPlaying) {
                playIcon.hide();
                pauseIcon.show();
            } else {
                playIcon.show();
                pauseIcon.hide();
            }
        }

        updateProgress() {
            if (!this.audio || !this.audio.duration) {
                return;
            }

            const current = this.audio.currentTime;
            const duration = this.audio.duration;
            const percentage = (current / duration) * 100;

            this.element.find('.gistmag-tts-progress-fill').css('width', percentage + '%');
            this.element.find('.gistmag-tts-current-time').text(this.formatTime(current));
        }

        updateDuration(duration) {
            this.element.find('.gistmag-tts-duration').text(this.formatTime(duration));
        }

        updateStatus(text) {
            this.element.find('.gistmag-tts-status-text').text(text);
        }

        showError(message) {
            const errorEl = this.element.find('.gistmag-tts-error');
            if (message) {
                errorEl.text(message).show();
            } else {
                errorEl.hide();
            }
        }

        resetProgress() {
            this.element.find('.gistmag-tts-progress-fill').css('width', '0%');
            this.element.find('.gistmag-tts-current-time').text('0:00');
        }

        formatTime(seconds) {
            if (isNaN(seconds)) {
                return '0:00';
            }
            
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return mins + ':' + (secs < 10 ? '0' : '') + secs;
        }
    }

    // Initialize players when DOM is ready
    $(document).ready(function() {
        $('.gistmag-tts-player').each(function() {
            new GistMagTTSPlayer(this);
        });
    });

})(jQuery);

