<?php
/**
 * Main plugin class for GistMag TTS Reader
 *
 * @package GistMag_TTS_Reader
 */

if (!defined('ABSPATH')) {
    exit;
}

class GistMag_TTS_Reader {
    private $api;

    public function __construct() {
        $this->api = new GistMag_TTS_Reader_API();
    }

    public function run() {
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Add TTS player to posts
        if (get_option('gistmag_tts_reader_auto_show', true)) {
            $position = get_option('gistmag_tts_reader_position', 'top');
            if ($position === 'top') {
                add_filter('the_content', array($this, 'add_tts_player_top'), 10);
            } else {
                add_filter('the_content', array($this, 'add_tts_player_bottom'), 20);
            }
        }
        
        // Add shortcode for manual placement
        add_shortcode('gistmag_tts', array($this, 'tts_player_shortcode'));
        
        // AJAX endpoint for streaming TTS
        add_action('wp_ajax_gistmag_tts_stream', array($this, 'ajax_stream_tts'));
        add_action('wp_ajax_nopriv_gistmag_tts_stream', array($this, 'ajax_stream_tts'));
        
        // AJAX endpoint for getting voices (frontend)
        add_action('wp_ajax_gistmag_tts_get_voices_frontend', array($this, 'ajax_get_voices_frontend'));
        add_action('wp_ajax_nopriv_gistmag_tts_get_voices_frontend', array($this, 'ajax_get_voices_frontend'));
    }

    public function enqueue_scripts() {
        if (!is_singular() && !is_home() && !is_archive()) {
            return; // Only load on single posts/pages
        }

        wp_enqueue_script(
            'gistmag-tts-reader',
            GISTMAG_TTS_READER_PLUGIN_URL . 'assets/gistmag-tts-reader.js',
            array('jquery'),
            GISTMAG_TTS_READER_VERSION,
            true
        );

        wp_enqueue_style(
            'gistmag-tts-reader',
            GISTMAG_TTS_READER_PLUGIN_URL . 'assets/gistmag-tts-reader.css',
            array(),
            GISTMAG_TTS_READER_VERSION
        );

        // Localize script with settings
        wp_localize_script('gistmag-tts-reader', 'gistmagTTSReader', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'apiBaseUrl' => GISTMAG_TTS_READER_API_BASE_URL,
            'nonce' => wp_create_nonce('gistmag_tts_reader_nonce'),
            'language' => get_option('gistmag_tts_reader_language', 'en'),
            'voiceName' => get_option('gistmag_tts_reader_voice_name', ''),
            'speed' => floatval(get_option('gistmag_tts_reader_speed', 1.0)),
            'getVoicesNonce' => wp_create_nonce('gistmag_tts_reader_get_voices'),
        ));
    }

    public function add_tts_player_top($content) {
        if (!is_singular() || !get_option('gistmag_tts_reader_enabled', true)) {
            return $content;
        }

        // Check display_on setting
        if (!$this->should_display_player()) {
            return $content;
        }

        $player_html = $this->get_tts_player_html();
        return $player_html . $content;
    }

    public function add_tts_player_bottom($content) {
        if (!is_singular() || !get_option('gistmag_tts_reader_enabled', true)) {
            return $content;
        }

        // Check display_on setting
        if (!$this->should_display_player()) {
            return $content;
        }

        $player_html = $this->get_tts_player_html();
        return $content . $player_html;
    }

    /**
     * Check if player should be displayed based on display_on setting
     */
    private function should_display_player() {
        $display_on = get_option('gistmag_tts_reader_display_on', 'both');
        
        if ($display_on === 'both') {
            return true; // Show on both posts and pages
        }
        
        if ($display_on === 'posts') {
            return is_single(); // Only show on single posts
        }
        
        if ($display_on === 'pages') {
            return is_page(); // Only show on pages
        }
        
        return true; // Default to showing on both
    }

    public function tts_player_shortcode($atts) {
        if (!get_option('gistmag_tts_reader_enabled', true)) {
            return '';
        }

        return $this->get_tts_player_html();
    }

    private function get_tts_player_html() {
        $post_id = get_the_ID();
        $post_title = get_the_title();
        $player_type = get_option('gistmag_tts_reader_player_type', 'full');
        $is_full_player = ($player_type === 'full');
        
        ob_start();
        ?>
        <div class="gistmag-tts-player" data-post-id="<?php echo esc_attr($post_id); ?>" data-player-type="<?php echo esc_attr($player_type); ?>">
            <div class="gistmag-tts-player-header">
                <h3 class="gistmag-tts-player-title">
                    <span class="gistmag-tts-icon">🎙️</span>
                    Listen to this post: <?php echo esc_html($post_title); ?>
                    <?php if ($is_full_player): ?>
                        <button type="button" class="gistmag-tts-settings-toggle" aria-label="Toggle settings" title="Settings">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                                <path d="M19.14,12.94c0.04-0.3,0.06-0.61,0.06-0.94c0-0.32-0.02-0.64-0.07-0.94l2.03-1.58c0.18-0.14,0.23-0.41,0.12-0.61 l-1.92-3.32c-0.12-0.22-0.37-0.29-0.59-0.22l-2.39,0.96c-0.5-0.38-1.03-0.7-1.62-0.94L14.4,2.81c-0.04-0.24-0.24-0.41-0.48-0.41 h-3.84c-0.24,0-0.43,0.17-0.47,0.41L9.25,5.35C8.66,5.59,8.12,5.92,7.63,6.29L5.24,5.33c-0.22-0.08-0.47,0-0.59,0.22L2.74,8.87 C2.62,9.08,2.66,9.34,2.86,9.48l2.03,1.58C4.84,11.36,4.8,11.69,4.8,12s0.02,0.64,0.07,0.94l-2.03,1.58 c-0.18,0.14-0.23,0.41-0.12,0.61l1.92,3.32c0.12,0.22,0.37,0.29,0.59,0.22l2.39-0.96c0.5,0.38,1.03,0.7,1.62,0.94l0.36,2.54 c0.05,0.24,0.24,0.41,0.48,0.41h3.84c0.24,0,0.44-0.17,0.47-0.41l0.36-2.54c0.59-0.24,1.13-0.56,1.62-0.94l2.39,0.96 c0.22,0.08,0.47,0,0.59-0.22l1.92-3.32c0.12-0.22,0.07-0.47-0.12-0.61L19.14,12.94z M12,15.6c-1.98,0-3.6-1.62-3.6-3.6 s1.62-3.6,3.6-3.6s3.6,1.62,3.6,3.6S13.98,15.6,12,15.6z"/>
                            </svg>
                        </button>
                    <?php endif; ?>
                </h3>
            </div>
            <div class="gistmag-tts-player-controls">
                <button class="gistmag-tts-play-btn" aria-label="Play audio">
                    <span class="gistmag-tts-play-icon">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M8 5v14l11-7z"/>
                        </svg>
                    </span>
                    <span class="gistmag-tts-pause-icon" style="display: none;">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M6 4h4v16H6V4zm8 0h4v16h-4V4z"/>
                        </svg>
                    </span>
                </button>
                <div class="gistmag-tts-progress-container">
                    <div class="gistmag-tts-progress-bar">
                        <div class="gistmag-tts-progress-fill"></div>
                    </div>
                    <div class="gistmag-tts-time">
                        <span class="gistmag-tts-current-time">0:00</span> / 
                        <span class="gistmag-tts-duration">--:--</span>
                    </div>
                </div>
                <div class="gistmag-tts-status">
                    <span class="gistmag-tts-status-text">Ready to play</span>
                </div>
            </div>
            <?php if ($is_full_player): ?>
            <div class="gistmag-tts-settings-panel" style="display: none;">
                <div class="gistmag-tts-setting-item" style="margin-top: 0;">
                    <label for="gistmag-tts-speed-control">
                        <span class="gistmag-tts-setting-label">Speed:</span>
                        <span class="gistmag-tts-speed-value">1.0x</span>
                    </label>
                    <input 
                        type="range" 
                        id="gistmag-tts-speed-control" 
                        class="gistmag-tts-speed-slider" 
                        min="0.25" 
                        max="2.0" 
                        step="0.25" 
                        value="1.0"
                    />
                    <div class="gistmag-tts-speed-labels">
                        <span>0.25x</span>
                        <span>0.5x</span>
                        <span>0.75x</span>
                        <span>1.0x</span>
                        <span>1.25x</span>
                        <span>1.5x</span>
                        <span>1.75x</span>
                        <span>2.0x</span>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            <div class="gistmag-tts-error" style="display: none;"></div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function ajax_stream_tts() {
        check_ajax_referer('gistmag_tts_reader_nonce', 'nonce');

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        if (!$post_id) {
            wp_send_json_error(array('message' => 'Invalid post ID'));
            return;
        }

        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(array('message' => 'Post not found'));
            return;
        }

        // Get post content and strip HTML
        $content = $post->post_content;
        $content = wp_strip_all_tags($content);
        $content = strip_shortcodes($content);
        $content = trim($content);

        if (empty($content)) {
            wp_send_json_error(array('message' => 'Post has no content'));
            return;
        }

        // Get settings
        $api_key = get_option('gistmag_tts_reader_api_key');
        if (empty($api_key)) {
            wp_send_json_error(array('message' => 'API key not configured. Please configure it in the plugin settings.'));
            return;
        }

        $language = get_option('gistmag_tts_reader_language', 'en');
        $voice_name = get_option('gistmag_tts_reader_voice_name', '');
        // Allow speed to be overridden by user input
        $speed = isset($_POST['speed']) ? floatval($_POST['speed']) : floatval(get_option('gistmag_tts_reader_speed', 1.0));
        $speed = max(0.25, min(2.0, $speed)); // Clamp between 0.25 and 2.0

        // Call the API to get streaming URL
        $stream_url = $this->api->get_stream_url($api_key, $content, $language, $voice_name, $speed);
        
        if ($stream_url === false) {
            wp_send_json_error(array('message' => 'Failed to initialize TTS stream'));
            return;
        }

        wp_send_json_success(array(
            'stream_url' => $stream_url,
            'post_id' => $post_id,
        ));
    }

    /**
     * AJAX handler to get voices for frontend player
     */
    public function ajax_get_voices_frontend() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'gistmag_tts_reader_get_voices')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }

        $language = isset($_POST['language']) ? sanitize_text_field(wp_unslash($_POST['language'])) : 'en';

        // Fetch voices from API
        $url = GISTMAG_TTS_READER_API_BASE_URL . '/tts/voices';
        if (!empty($language)) {
            $url .= '?language=' . urlencode($language);
        }

        $response = wp_remote_get($url, array(
            'timeout' => 10,
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => 'Failed to fetch voices: ' . $response->get_error_message()));
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            wp_send_json_error(array('message' => 'Failed to fetch voices. HTTP ' . $status_code));
            return;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!isset($data['voices']) || !is_array($data['voices'])) {
            wp_send_json_error(array('message' => 'Invalid response from API'));
            return;
        }

        wp_send_json_success(array('voices' => $data['voices']));
    }
}

