<?php

class GistMag_TTS_Reader_Admin {
    private $api;

    public function __construct() {
        $this->api = new GistMag_TTS_Reader_API();
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_gistmag_tts_proxy', array($this, 'ajax_proxy_stream'));
        add_action('wp_ajax_nopriv_gistmag_tts_proxy', array($this, 'ajax_proxy_stream'));
    }

    public function add_admin_menu() {
        add_options_page(
            'GistMag TTS Reader Settings',
            'TTS Reader',
            'manage_options',
            'gistmag-tts-reader',
            array($this, 'render_settings_page')
        );
    }

    public function register_settings() {
        // phpcs:disable PluginCheck.CodeAnalysis.SettingSanitization.register_settingMissing
        // All settings have sanitize_callback defined in the array parameter
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_api_key', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_enabled', array('sanitize_callback' => 'rest_sanitize_boolean'));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_language', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_voice_name', array('sanitize_callback' => 'sanitize_text_field'));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_speed', array('sanitize_callback' => array($this, 'sanitize_speed')));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_auto_show', array('sanitize_callback' => 'rest_sanitize_boolean'));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_position', array('sanitize_callback' => array($this, 'sanitize_position')));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_player_type', array('sanitize_callback' => array($this, 'sanitize_player_type')));
        register_setting('gistmag_tts_reader_settings', 'gistmag_tts_reader_display_on', array('sanitize_callback' => array($this, 'sanitize_display_on')));
        // phpcs:enable
    }

    public function sanitize_speed($value) {
        $speed = floatval($value);
        return max(0.25, min(4.0, $speed));
    }

    public function sanitize_position($value) {
        $allowed = array('top', 'bottom');
        return in_array($value, $allowed, true) ? $value : 'top';
    }

    public function sanitize_player_type($value) {
        $allowed = array('full', 'simple');
        return in_array($value, $allowed, true) ? $value : 'full';
    }

    public function sanitize_display_on($value) {
        $allowed = array('posts', 'pages', 'both');
        return in_array($value, $allowed, true) ? $value : 'both';
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Handle form submission
        if (isset($_POST['gistmag_tts_reader_submit']) && check_admin_referer('gistmag_tts_reader_settings')) {
            // Get and sanitize POST values
            $api_key = isset($_POST['gistmag_tts_reader_api_key']) ? sanitize_text_field(wp_unslash($_POST['gistmag_tts_reader_api_key'])) : '';
            $enabled = isset($_POST['gistmag_tts_reader_enabled']);
            $language = isset($_POST['gistmag_tts_reader_language']) ? sanitize_text_field(wp_unslash($_POST['gistmag_tts_reader_language'])) : 'en';
            $voice_name = isset($_POST['gistmag_tts_reader_voice_name']) ? sanitize_text_field(wp_unslash($_POST['gistmag_tts_reader_voice_name'])) : '';
            $speed = isset($_POST['gistmag_tts_reader_speed']) ? floatval(wp_unslash($_POST['gistmag_tts_reader_speed'])) : 1.0;
            $auto_show = isset($_POST['gistmag_tts_reader_auto_show']);
            $position = isset($_POST['gistmag_tts_reader_position']) ? sanitize_text_field(wp_unslash($_POST['gistmag_tts_reader_position'])) : 'top';
            $player_type = isset($_POST['gistmag_tts_reader_player_type']) ? sanitize_text_field(wp_unslash($_POST['gistmag_tts_reader_player_type'])) : 'full';
            $display_on = isset($_POST['gistmag_tts_reader_display_on']) ? sanitize_text_field(wp_unslash($_POST['gistmag_tts_reader_display_on'])) : 'both';
            
            update_option('gistmag_tts_reader_api_key', $api_key);
            update_option('gistmag_tts_reader_enabled', $enabled);
            update_option('gistmag_tts_reader_language', $language);
            update_option('gistmag_tts_reader_voice_name', $voice_name);
            update_option('gistmag_tts_reader_speed', $speed);
            update_option('gistmag_tts_reader_auto_show', $auto_show);
            update_option('gistmag_tts_reader_position', $position);
            update_option('gistmag_tts_reader_player_type', $player_type);
            update_option('gistmag_tts_reader_display_on', $display_on);
            
            echo '<div class="notice notice-success"><p>' . esc_html__('Settings saved!', 'gistmag-tts-reader') . '</p></div>';
        }

        $api_key = get_option('gistmag_tts_reader_api_key', '');
        $enabled = get_option('gistmag_tts_reader_enabled', true);
        $language = get_option('gistmag_tts_reader_language', 'en');
        $voice_name = get_option('gistmag_tts_reader_voice_name', '');
        $speed = get_option('gistmag_tts_reader_speed', 1.0);
        $auto_show = get_option('gistmag_tts_reader_auto_show', true);
        $position = get_option('gistmag_tts_reader_position', 'top');
        $player_type = get_option('gistmag_tts_reader_player_type', 'full');
        $display_on = get_option('gistmag_tts_reader_display_on', 'both');

        // Verify API key if provided
        $api_key_valid = false;
        $api_key_message = '';
        if (!empty($api_key)) {
            $api_key_valid = $this->api->verify_api_key($api_key);
            $api_key_message = $api_key_valid ? '✓ API key is valid' : '✗ API key is invalid';
        }

        ?>
        <div class="wrap">
            <h1>GistMag TTS Reader Settings</h1>
            
            <form method="post" action="">
                <?php wp_nonce_field('gistmag_tts_reader_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_api_key">API Key</label>
                        </th>
                        <td>
                            <input 
                                type="text" 
                                id="gistmag_tts_reader_api_key" 
                                name="gistmag_tts_reader_api_key" 
                                value="<?php echo esc_attr($api_key); ?>" 
                                class="regular-text"
                                placeholder="Enter your GistMag API key"
                            />
                            <p class="description">
                                Get your API key from <a href="https://gistmag.co.uk/dashboard/api-key" target="_blank">your dashboard</a>.
                                <?php if (!empty($api_key)): ?>
                                    <br><strong style="color: <?php echo $api_key_valid ? 'green' : 'red'; ?>;">
                                        <?php echo esc_html($api_key_message); ?>
                                    </strong>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_enabled">Enable TTS Reader</label>
                        </th>
                        <td>
                            <input 
                                type="checkbox" 
                                id="gistmag_tts_reader_enabled" 
                                name="gistmag_tts_reader_enabled" 
                                value="1" 
                                <?php checked($enabled, true); ?>
                            />
                            <label for="gistmag_tts_reader_enabled">Show TTS player on posts</label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_auto_show">Auto Show Player</label>
                        </th>
                        <td>
                            <input 
                                type="checkbox" 
                                id="gistmag_tts_reader_auto_show" 
                                name="gistmag_tts_reader_auto_show" 
                                value="1" 
                                <?php checked($auto_show, true); ?>
                            />
                            <label for="gistmag_tts_reader_auto_show">Automatically show player</label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_display_on">Display On</label>
                        </th>
                        <td>
                            <select id="gistmag_tts_reader_display_on" name="gistmag_tts_reader_display_on">
                                <option value="posts" <?php selected($display_on, 'posts'); ?>>Posts Only</option>
                                <option value="pages" <?php selected($display_on, 'pages'); ?>>Pages Only</option>
                                <option value="both" <?php selected($display_on, 'both'); ?>>Both Posts and Pages</option>
                            </select>
                            <p class="description">Choose where to display the TTS player: posts only, pages only, or both.</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_position">Player Position</label>
                        </th>
                        <td>
                            <select id="gistmag_tts_reader_position" name="gistmag_tts_reader_position">
                                <option value="top" <?php selected($position, 'top'); ?>>Top</option>
                                <option value="bottom" <?php selected($position, 'bottom'); ?>>Bottom</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_language">Language</label>
                        </th>
                        <td>
                            <select id="gistmag_tts_reader_language" name="gistmag_tts_reader_language">
                                <option value="en" <?php selected($language, 'en'); ?>>English (US)</option>
                                <option value="es" <?php selected($language, 'es'); ?>>Spanish</option>
                                <option value="fr" <?php selected($language, 'fr'); ?>>French</option>
                                <option value="de" <?php selected($language, 'de'); ?>>German</option>
                                <option value="it" <?php selected($language, 'it'); ?>>Italian</option>
                                <option value="pt" <?php selected($language, 'pt'); ?>>Portuguese</option>
                                <option value="ja" <?php selected($language, 'ja'); ?>>Japanese</option>
                                <option value="ko" <?php selected($language, 'ko'); ?>>Korean</option>
                                <option value="zh" <?php selected($language, 'zh'); ?>>Chinese</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_voice_name">Voice Name (Optional)</label>
                        </th>
                        <td>
                            <input 
                                type="text" 
                                id="gistmag_tts_reader_voice_name" 
                                name="gistmag_tts_reader_voice_name" 
                                value="<?php echo esc_attr($voice_name); ?>" 
                                class="regular-text"
                                placeholder="Leave empty for auto-select"
                            />
                            <p class="description">
                                Specify a voice name (e.g., en-US-Wavenet-D) or leave empty for auto-selection.
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_speed">Default Speed</label>
                        </th>
                        <td>
                            <input 
                                type="number" 
                                id="gistmag_tts_reader_speed" 
                                name="gistmag_tts_reader_speed" 
                                value="<?php echo esc_attr($speed); ?>" 
                                min="0.25" 
                                max="4.0" 
                                step="0.1"
                            />
                            <p class="description">Default speech speed multiplier (0.25x to 4.0x, default: 1.0x). Users can adjust this in the player if full media mode is enabled.</p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="gistmag_tts_reader_player_type">Player Type</label>
                        </th>
                        <td>
                            <select id="gistmag_tts_reader_player_type" name="gistmag_tts_reader_player_type">
                                <option value="full" <?php selected($player_type, 'full'); ?>>Full Media Player (with speed control and settings)</option>
                                <option value="simple" <?php selected($player_type, 'simple'); ?>>Simple Player (basic controls only)</option>
                            </select>
                            <p class="description">Choose between a full-featured player with user controls or a simple player with basic play/pause functionality.</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('Save Settings', 'primary', 'gistmag_tts_reader_submit'); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Proxy stream endpoint for AJAX
     * This handles the actual streaming from the API
     */
    public function ajax_proxy_stream() {
        // Handle CORS preflight
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type');
            header('Access-Control-Max-Age: 86400');
            exit(0);
        }

        // Verify nonce - use GET parameter for post_id
        $post_id = isset($_GET['post_id']) ? intval($_GET['post_id']) : 0;
        if (!$post_id) {
            header('Content-Type: application/json');
            wp_send_json_error(array('message' => 'Invalid post ID'));
            return;
        }

        // For non-logged-in users, we still verify nonce but it's less strict
        // The nonce is generated per page load, so it should work
        $nonce = '';
        if (isset($_GET['nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_GET['nonce']));
        }
        if (empty($nonce) || !wp_verify_nonce($nonce, 'gistmag_tts_reader_nonce')) {
            header('Content-Type: application/json');
            wp_send_json_error(array('message' => 'Invalid nonce'));
            return;
        }

        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error(array('message' => 'Post not found'));
            return;
        }

        $content = wp_strip_all_tags($post->post_content);
        $content = strip_shortcodes($content);
        $content = trim($content);

        if (empty($content)) {
            wp_send_json_error(array('message' => 'Post has no content'));
            return;
        }

        $api_key = get_option('gistmag_tts_reader_api_key');
        if (empty($api_key)) {
            wp_send_json_error(array('message' => 'API key not configured'));
            return;
        }

        $language = get_option('gistmag_tts_reader_language', 'en');
        $voice_name = get_option('gistmag_tts_reader_voice_name', '');
        // Allow speed to be overridden by user input
        $speed = isset($_GET['speed']) ? floatval($_GET['speed']) : floatval(get_option('gistmag_tts_reader_speed', 1.0));
        $speed = max(0.25, min(2.0, $speed)); // Clamp between 0.25 and 2.0

        // Use cURL directly for streaming
        // Note: wp_remote_get() doesn't support streaming, so cURL is required here
        // phpcs:ignore WordPress.WP.AlternativeFunctions.curl_curl_init
        $url = GISTMAG_TTS_READER_API_BASE_URL . '/tts/stream';
        
        $body = array(
            'api_key' => $api_key,
            'text' => $content,
            'language' => $language,
            'chunk_size' => 200,
            'speed' => $speed,
            'source' => 'wordpress',
        );

        if (!empty($voice_name)) {
            $body['voice_name'] = $voice_name;
        }

        // Clear any previous output
        if (ob_get_level() > 0) {
            ob_end_clean();
        }

        // Set headers for streaming
        header('Content-Type: audio/mpeg');
        header('Cache-Control: no-cache, must-revalidate');
        header('X-Accel-Buffering: no'); // Disable buffering for nginx
        header('Connection: keep-alive');
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type');

        // Use cURL for streaming (wp_remote_get doesn't support streaming)
        // phpcs:disable WordPress.WP.AlternativeFunctions.curl_curl_init, WordPress.WP.AlternativeFunctions.curl_curl_setopt, WordPress.WP.AlternativeFunctions.curl_curl_exec, WordPress.WP.AlternativeFunctions.curl_curl_getinfo, WordPress.WP.AlternativeFunctions.curl_curl_error, WordPress.WP.AlternativeFunctions.curl_curl_close
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/json',
        ));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, false);
        curl_setopt($ch, CURLOPT_WRITEFUNCTION, function($ch, $data) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Audio binary data should not be escaped
            echo $data;
            flush();
            if (ob_get_level() > 0) {
                ob_flush();
            }
            return strlen($data);
        });
        curl_setopt($ch, CURLOPT_TIMEOUT, 300); // 5 minutes
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

        $result = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        // phpcs:enable

        if ($http_code !== 200 || !empty($curl_error)) {
            // Error occurred - but we can't send JSON here as headers are already sent
            // The audio element will handle the error
            if (defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log')) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Only in debug mode
                error_log('GistMag TTS Proxy Error: HTTP ' . $http_code . ' - ' . $curl_error);
            }
            exit;
        }

        exit;
    }
}

