<?php

class GistMag_TTS_Reader_API {
    private $api_base_url;

    public function __construct() {
        $this->api_base_url = GISTMAG_TTS_READER_API_BASE_URL;
    }

    /**
     * Get streaming URL for TTS
     * Since the API streams directly, we create a proxy endpoint
     */
    public function get_stream_url($api_key, $text, $language = 'en', $voice_name = '', $speed = 1.0) {
        // Create a proxy URL that WordPress can use
        // The actual streaming will be handled by JavaScript
        $nonce = wp_create_nonce('gistmag_tts_stream_' . md5($text));
        
        // Store the request parameters temporarily (in a more production setup, use transients)
        // For now, we'll pass them via the URL and handle in JavaScript
        return add_query_arg(array(
            'action' => 'gistmag_tts_proxy',
            'nonce' => $nonce,
        ), admin_url('admin-ajax.php'));
    }

    /**
     * Verify API key
     */
    public function verify_api_key($api_key) {
        if (empty($api_key)) {
            return false;
        }

        $url = $this->api_base_url . '/tts/voices'; // Use a simple endpoint to verify
        
        // Actually, we should use the verify endpoint from Supabase
        // But for now, we'll verify by attempting a small TTS request
        $verify_url = 'https://eutekssvzhtkwzperkzb.supabase.co/functions/v1/verify-key-public';
        
        $response = wp_remote_post($verify_url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'api_key' => $api_key,
            )),
            'timeout' => 10,
        ));

        if (is_wp_error($response)) {
            return false;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return $status_code === 200 && isset($data['valid']) && $data['valid'] === true;
    }

    /**
     * Proxy TTS stream request
     * This handles the actual API call and streams the response
     */
    public function proxy_stream($api_key, $text, $language = 'en', $voice_name = '', $speed = 1.0, $chunk_size = 200) {
        $url = $this->api_base_url . '/tts/stream';
        
        $body = array(
            'api_key' => $api_key,
            'text' => $text,
            'language' => $language,
            'chunk_size' => $chunk_size,
            'speed' => $speed,
            'source' => 'wordpress',
        );

        if (!empty($voice_name)) {
            $body['voice_name'] = $voice_name;
        }

        $response = wp_remote_post($url, array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode($body),
            'timeout' => 300, // 5 minutes for long posts
            'stream' => true,
        ));

        if (is_wp_error($response)) {
            return false;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            $body = wp_remote_retrieve_body($response);
            $error_data = json_decode($body, true);
            return array(
                'error' => true,
                'message' => isset($error_data['error']) ? $error_data['error'] : 'Failed to generate TTS',
                'status_code' => $status_code,
            );
        }

        return $response;
    }
}

